-- Author:Raptor5
-- Name:R5Utils
-- Description: Function-Collection
-- Icon:
-- Hide: yes
-- Version: 1.0.3

R5Utils = {}

-- GENERAL
------------------------------------------------------------------

--[[ recursive search function; iterrate trough full child structure from given startNode
-- params:	
	startNode:	index of parentnode [int]
	name:		name of searched element [string]
-- return:
	index:		index of element if found
]]
function R5Utils.searchNodeByName(startNode, name)
	if getNumOfChildren(startNode) > 0 then
		local index = getChild(startNode, name)
	
		if index > 0 then return index end
		
		for i=0, getNumOfChildren(startNode), 1 do
			index = R5Utils.searchNodeByName(getChildAt(startNode, i), name);
			if index > 0 then return index end
		end
	end
	return 0
end

--[[ recursive search function; iterrate trough full child structure from given startNode
-- params:	
	startNode:			index of parentnode [int]
	name:				name of searched element or pattern [string]
	isCaseSensitive:	search case sensitive [bool]
	searchExact:		true = match full name; false = match name pattern [bool]
-- return:
	a:					table of matching node-ids [table]
]]
function R5Utils.searchAllNodesByName(startNode, name, isCaseSensitive, searchExact)
	local a = {}
	nName = isCaseSensitive and getName(startNode) or getName(startNode):lower()
	name = isCaseSensitive and name or name:lower()

	if searchExact then
		if name == nName then
			table.insert(a, startNode)
		end
	elseif string.find(nName, name) then
        table.insert(a, startNode)
	end
		
	if getNumOfChildren(startNode) > 0 then
		for i=0, getNumOfChildren(startNode)-1, 1 do
			local b = R5Utils.searchAllNodesByName(getChildAt(startNode, i), name, isCaseSensitive, searchExact)
            if table.getn(b) > 0 then 
                for k,v in pairs(b) do 
                    if not tableContains(a, v) then table.insert(a,v) end
                end
            end
		end
	end
	return a
end

--[[ helperfunction, checks if table/array contains specific value
-- params:	
	tab:	    table/array
	val:		searched value
-- return:
	[bool]:		value found=true not-found=false
]]
function tableContains(tab, val)
   for _,v in pairs(tab) do
      if v == val then
        return true
      end
    end 
    return false
end


--[[ get nearest point from list
-- params:	
	node:	    startNode id [int]
	tab:		Node list [table int]
-- return:
	nodeId:		ID of nearest Node [int]
	minDistance:distance to nearest node [float]	
]]
function R5Utils.getNearestNode(x,y,z, tab)
	local minDistance = math.huge -- +infinity
	local nodeId = nil
	
	for _,v in pairs(tab) do 
        local a,b,c = getWorldTranslation(v)
		local d = R5Utils.getPointDisance3D(x,y,z, a,b,c)
		if d < minDistance then
			minDistance = d
			nodeId = v
		end
	end
	
	return nodeId, minDistance
end

-- MATH
------------------------------------------------------------------

--[[ returns distance between two points in 3D-space
-- params:	
	x1,y1,z1:	xyz Coordinates P1
	x2,y2,z2:	xyz Coordinates P2
-- return:
	distance:	distance between points [float]
]]
function R5Utils.getPointDisance3D(x1,y1,z1, x2,y2,z2)
	return ( (x2-x1)^2 + (y2-y1)^2 + (z2-z1)^2 )^0.5
end


--[[ calculate normalized direction vector from given rotation angles (xyz)
	 +xAxis = forward direction
-- params:	
	rx, ry, rz:	xyz rotations from Point/Transform in rad [float]
-- return:
	dx, dy, dz:	direction-vector [float]
]]
function R5Utils.rotationToVector(rx,ry,rz)
	local dx,dy,dz
	ry = (rx < 1 and rz < 1) and -ry or ry --fix negative rotations
	
	dx = math.cos(ry) * math.cos(rz)
	dy = math.sin(rz)
	dz = math.sin(ry) * math.cos(rz)
	
	return dx, dy, dz
end

--[[ calculate normalized direction vector from given nodeId
	 +xAxis = forward direction
-- params:	
	nodeId:		Transform-ID [int]
-- return:
	dx, dy, dz:	direction-vector [float]
]]
function R5Utils.nodeRotationToVector(nodeId)
	local rx,ry,rz = getWorldRotation(nodeId)
	return R5Utils.rotationToVector(rx,ry,rz)
end

--[[ get List of all direct Child-elements from given Node
-- params:
    nodeId: ParentNode
--return:
    List of ChildIds
]]
function R5Utils.getChilds(nodeId)
    local numChilds = getNumOfChildren(nodeId)
    local childs = {}

    for i=0, (numChilds-1) do
        table.insert(childs, getChildAt(nodeId, i)) 
    end
    return childs
end

--[[ get List of all Child-elements and subChilds from given Node
-- params:
    nodeId: ParentNode
--return:
    List of ChildIds
]]
function R5Utils.getAllChilds(nodeId)
    local numChilds = getNumOfChildren(nodeId)
    local childs = {}

    for i=0, (numChilds-1) do
        local b = R5Utils.getAllChilds(getChildAt(nodeId, i))
        if table.getn(b) > 0 then 
            for k,v in pairs(b) do 
                if not tableContains(childs, v) then table.insert(childs,v) end
            end
        end 
    end
    table.insert(childs, nodeId)
    return childs
end

--[[ get Terrain-Node ID
-- params:
--return:
    TerrainID [int]
]]
function R5Utils.getTerrainRootNode()
	local terrain = getChild(getChildAt(getRootNode(), 0), "terrain")
	if terrain > 0 then
		return terrain
	end
	error("Terrain Node not Found!")
	return -1
end

--[[ Get Sign of Number Value
-- params:
    x: 		number [float]
--return:
    -1 :	negative number
	 1 :	positive number
]]
function R5Utils.sign(x)
    return (x > 0 and 1) or (x == 0 and 0) or -1
end

--[[			CHANGELOG
===================================================
V 1.0.3:
+	added function sign
V 1.0.2:
+	added function getTerrainRootNode
V 1.0.1:
+	added function getAllChilds 
+   added function getChilds 
]]
