-- Author:Raptor5
-- Name:R5FoliageUtils
-- Description:
-- Icon:
-- Hide: yes
-- Version: 1.0.0

R5FoliageUtils = {}

-- INCLUDES
source("GES_R5Utils.lua")
source("editorUtils.lua")

-- VARIABLES 
------------------------------------------------------------------
local NUM_BITS_GDM = 12

local FRUIT_DENSITY = nil
local GROUND_DENSITY = nil

local TERRAIN = nil

local MMODIFIER_DEL_ALL = nil


local LAYER_NAMES = {
	-- detailLayers:
	"terrainDetail",
	-- fruitLayers:
	"meadow", "grass", "wheat", "canola", "barley", "maize", "potato", "sugarBeet", "sunflower",
	"soybean", "oilseedRadish", "poplar", "oat", "sugarCane", "cotton", "sorghum", "olive", "grape",
	"meadowDLC", "meadowFR",
	-- decoLayers:
	"decoFoliageUS", "groundFoliage", "decoBush", "decoFoliageFR", "decoFoliageAlpine"
}

-- FUNCTIONS
------------------------------------------------------------------

--[[ Load DensityMaps from File
-- params:	
	fruitDensityPath:	filePath of FruitDensity.gdm 	[string]
	groundDensityPath:	filePath of GroundtDensity.gdm 	[string]
-- return:
    void
]]
function R5FoliageUtils.loadDensityMaps(fruitDensityPath, groundDensityPath, forceReload)
	if FRUIT_DENSITY ~= nil and GROUND_DENSITY ~= nil and not forceReload then
		print("Density-Maps already loaded. - continuing ...")
		print("set forceReload = true for overwrite")
	end
	
	print("[i]: loading Density-Maps")
    FRUIT_DENSITY = createBitVectorMap("FruitDensity")
	if not loadBitVectorMapFromFile(FRUIT_DENSITY, fruitDensityPath, NUM_BITS_GDM) then
		print("[e]: Failed to load FruitDensity! - aborting!")
		return;
	end;
	
	GROUND_DENSITY = createBitVectorMap("GroundDensity")
	if not loadBitVectorMapFromFile(GROUND_DENSITY, groundDensityPath, NUM_BITS_GDM) then
		print("[e]: Failed to load GroundDensity! - aborting!")
		return;
	end;
	print("[i]: loading Density-Maps - done!")
end

--[[ Check if densitymaps are loaded
-- params:	
-- return:
    bool 	loaded = true else false
]]
function densityMapsLoaded()
	if GROUND_DENSITY == nil or FRUIT_DENSITY == nil then
		return false
	end
	return true
end

--[[ Get Densitymap Multi modifier for all Fruit and Detail Layers
-- params:	
	
-- return:
    DensityMapMultiModifier MMODIFIER_DEL_ALL [DensityMapMultiModifier]
]]
function R5FoliageUtils.getDeleteAllMultiModifier()
	if MMODIFIER_DEL_ALL ~= nil then
		return MMODIFIER_DEL_ALL
	else
		-- create new DensityMapMultiModifier 
		MMODIFIER_DEL_ALL = DensityMapMultiModifier.new()
		
		-- create modifier for all Layers
		for i,name in ipairs(LAYER_NAMES) do
			local modifier = R5FoliageUtils.getModifierByName(name)
			if modifier ~= nil then
				-- add modifiert to multimodifier
				MMODIFIER_DEL_ALL:addExecuteSet(0, modifier)
			end
		end
		
		return MMODIFIER_DEL_ALL
	end
end

--[[ Applying DensityMultiModifier on Selected Parallelogram defined by 3 Points
-- params:	
	multiModifier:		MultiModifier [DensityMapMultiModifier]
	sX:					StartCoord X (world-coord) [float]
	sZ:					StartCoord Z (world-coord) [float]
	wX:					WidthCoord X (world-coord) [float]
	wZ:					WidthCoord Z (world-coord) [float]
	hX:					HeightCoord X (world-coord) [float]
	hZ:					HeightCoord Z (world-coord) [float]
-- return:
	void
]]
function R5FoliageUtils.applyMultiModifier(multiModifier, sX, sZ, wX, wZ, hX, hZ)
	if not densityMapsLoaded() then
		print("[e]: DensityMaps not loaded. - aborting!")
		return
	end
	multiModifier:updateParallelogramWorldCoords(sX, sZ, wX, wZ, hX, hZ, DensityCoordType.POINT_POINT_POINT)
	multiModifier:execute(false)
end

--[[ Applying DensityModifier on Selected Parallelogram defined by 3 Points
-- params:	
	modifier:		MultiModifier [DensityMapModifier]
	state:			Foliage Grow-State ID [int]
	sX:				StartCoord X (world-coord) [float]
	sZ:				StartCoord Z (world-coord) [float]
	wX:				WidthCoord X (world-coord) [float]
	wZ:				WidthCoord Z (world-coord) [float]
	hX:				HeightCoord X (world-coord) [float]
	hZ:				HeightCoord Z (world-coord) [float]
-- return:
	void
]]
function R5FoliageUtils.applyModifier(modifier, state, sX, sZ, wX, wZ, hX, hZ)
	if not densityMapsLoaded() then
		print("[e]: DensityMaps not loaded. - aborting!")
		return
	end
	local terrSi = getTerrainSize(TERRAIN)
	modifier:setParallelogramUVCoords(sX /terrSi +0.5, sZ/terrSi +0.5, wX/terrSi +0.5, wZ/terrSi +0.5, hX/terrSi +0.5, hZ/terrSi +0.5, DensityCoordType.POINT_POINT_POINT)
	modifier:executeSet(state)
end

--[[ Applying DensityMultiModifier on Selected Parallelogram defined by 3 Points
-- params:	
	name:		Layername [string]
-- return:
	modifier:	Layer Modifier [DensityMapModifier]
]]
function R5FoliageUtils.getModifierByName(name)
	if TERRAIN == nil then
		TERRAIN = R5Utils.getTerrainRootNode()
		if TERRAIN < 1 then
			print("[e]: Terrain not found!")
			return nil
		end
	end
	
	local id = getTerrainDataPlaneByName(TERRAIN, name)
	if id > 0 then
		return DensityMapModifier.new(id, 0, NUM_BITS_GDM/2)
	else
		print("[i]: Layer "..name.." not found. continue...")
		return nil
	end
end

--[[ Clears all Foiliage Layers along Spline with specific width
-- params:	
	spline:		Spline ID [int]
	width:		clearing width [float]
-- return:
	void
]]
function R5FoliageUtils.clearAllBySpline(spline, width)
	if not densityMapsLoaded() then
		print("[e]: DensityMaps not loaded. - aborting!")
		return
	end
	
	local multiMod = R5FoliageUtils.getDeleteAllMultiModifier()
	
	local sLength = getSplineLength(spline)
	local currentPos = 0.0		-- current position on spline
	local stepDistMeter = 1 -- distance in meter
	local stepDist = stepDistMeter / sLength  -- relative size [0..1]
	
	local x, y, z 			-- world coords
	local rx, ry, rz		-- world vectors
	
	local offset = width / 2
	
	while currentPos <= 1.0 do
		-- get XYZ at position on spline
		local mPosX, mPosY, mPosZ = getSplinePosition(spline, currentPos)
		-- directional vector at the point
		local mDirX, mDirY, mDirZ = getSplineDirection(spline, currentPos)
		local mVecDx, mVecDy, mVecDz = EditorUtils.crossProduct(mDirX, mDirY, mDirZ, 0, 1, 0)

		local sX = mPosX + offset * mVecDx
		local sZ = mPosZ + offset * mVecDz
		local wX = mPosX  - offset * mVecDx
		local wZ = mPosZ  - offset * mVecDz

		local nextSP = currentPos + stepDist > 1 and 1 or currentPos+stepDist
		-- get next XYZ at position on spline 
		local nPosX, nPosY, nPosZ = getSplinePosition(spline, nextSP)

		-- directional vector at the next point
		local nDirX, nDirY, nDirZ = getSplineDirection (spline, nextSP)
		local nVecDx, nVecDy, nVecDz = EditorUtils.crossProduct(nDirX, nDirY, nDirZ, 0, 1, 0)

		local hX, hZ

		hX = nPosX - offset * nVecDx
		hZ = nPosZ - offset * nVecDz
		R5FoliageUtils.applyMultiModifier(multiMod, wX, wZ, sX, sZ, hX, hZ)

		hX = nPosX + offset * nVecDx
		hZ = nPosZ + offset * nVecDz
		R5FoliageUtils.applyMultiModifier(multiMod, sX, sZ, wX, wZ, hX, hZ)
		
		currentPos = currentPos + stepDist
	end
end

--[[ Draws Foliage along Spline with specific width
-- params:	
	spline:		Spline ID [int]
	width:		clearing width [float]
	layername: 	name of the FoliageLayer [string]
	state:		LayerState ID [int]
-- return:
	void
]]
function R5FoliageUtils.drawLayerBySpline(spline, width, layername, state)
	if not densityMapsLoaded() then
		print("[e]: DensityMaps not loaded. - aborting!")
		return
	end
	
	local modifier = R5FoliageUtils.getModifierByName(layername)
	if modifier == nil then
		print("Error: Layer "..layername.." not found - aborting!")
		return
	end
	
	local sLength = getSplineLength(spline)
	local currentPos = 0.0		-- current position on spline
	local stepDistMeter = 0.5 -- distance in meter
	local stepDist = stepDistMeter / sLength  -- relative size [0..1]
	
	local x, y, z 			-- world coords
	local rx, ry, rz		-- world vectors
	
	local offset = width / 2
	
	while currentPos <= 1.0 do
		-- get XYZ at position on spline
		local mPosX, mPosY, mPosZ = getSplinePosition(spline, currentPos)
		-- directional vector at the point
		local mDirX, mDirY, mDirZ = getSplineDirection(spline, currentPos)
		local mVecDx, mVecDy, mVecDz = EditorUtils.crossProduct(mDirX, mDirY, mDirZ, 0, 1, 0)

		local sX = mPosX + offset * mVecDx
		local sZ = mPosZ + offset * mVecDz
		local wX = mPosX  - offset * mVecDx
		local wZ = mPosZ  - offset * mVecDz

		local nextSP = currentPos + stepDist > 1 and 1 or currentPos+stepDist
		-- get next XYZ at position on spline 
		local nPosX, nPosY, nPosZ = getSplinePosition(spline, nextSP)

		-- directional vector at the next point
		local nDirX, nDirY, nDirZ = getSplineDirection (spline, nextSP)
		local nVecDx, nVecDy, nVecDz = EditorUtils.crossProduct(nDirX, nDirY, nDirZ, 0, 1, 0)

		local hX, hZ

		hX = nPosX - offset * nVecDx
		hZ = nPosZ - offset * nVecDz
		R5FoliageUtils.applyModifier(modifier, state, wX, wZ, sX, sZ, hX, hZ)

		hX = nPosX + offset * nVecDx
		hZ = nPosZ + offset * nVecDz
		R5FoliageUtils.applyModifier(modifier, state, sX, sZ, wX, wZ, hX, hZ)
		
		currentPos = currentPos + stepDist
	end

end
