--[[
    Author: PeterAH (Modding-Welt)
    Mod name: Air Compressor Sound
    Version: 1.0.0.0
    Date: April 2021
    Contact: https://www.modding-welt.com
    Discord: [MW] PeterAH#5807
]]

compressorSound = {}

local _modDirectory = g_currentModDirectory
local _compressorVolumeSetting = 1.00
local _compressorRunTime = 16
local _maxAirUsagePerFullStop = 17
local _trailerAttachAirUsagePercent = 26
local _debug = false

function compressorSound.prerequisitesPresent(specializations)
    return true
end

function compressorSound.registerEventListeners(vehicleType)
    SpecializationUtil.registerEventListener(vehicleType, "onPreLoad", compressorSound)
    SpecializationUtil.registerEventListener(vehicleType, "onPostLoad", compressorSound)
    SpecializationUtil.registerEventListener(vehicleType, "onStopMotor", compressorSound)
    SpecializationUtil.registerEventListener(vehicleType, "onUpdate", compressorSound)
    SpecializationUtil.registerEventListener(vehicleType, "onReadUpdateStream", compressorSound)
    SpecializationUtil.registerEventListener(vehicleType, "onWriteUpdateStream", compressorSound)
end

function compressorSound:onPreLoad(savegame)
    self.lastMaxSpeed = 22.5
    self.attachedAirUsage = 0
    self.attachedAirConsumptionTimer = 0
    self.streamDoRefill = false

    local modDescXMLFile = loadXMLFile("modDescXMLFile", _modDirectory .. "modDesc.xml")
    _compressorVolumeSetting = Utils.getNoNil(getXMLFloat(modDescXMLFile, "modDesc.compressorSound#volumeSetting"), 1)
    _compressorRunTime = Utils.getNoNil(getXMLFloat(modDescXMLFile, "modDesc.compressorSound#compressorRunTime"), 16)
    _maxAirUsagePerFullStop = Utils.getNoNil(getXMLFloat(modDescXMLFile, "modDesc.compressorSound#maxAirUsagePercentPerFullStop"), 17)
    _trailerAttachAirUsagePercent = Utils.getNoNil(getXMLFloat(modDescXMLFile, "modDesc.airAttachSound#trailerAttachAirUsagePercent"), 26)
    _debug = Utils.getNoNil(getXMLBool(modDescXMLFile, "modDesc.debugCompressorSound#showIngameDebugInfo"), false)

    i = 0
    self.hasCompressorSound = true
    local i3dFilename = getXMLString(self.xmlFile, "vehicle.base.filename")
    if string.find(i3dFilename, "/") == nil then
        i3dFilename = "/" .. i3dFilename
    end
    repeat
        local key = string.format("modDesc.excludeVehicles.noCompressorSound(%d)", i)
        if not hasXMLProperty(modDescXMLFile, key) then
            break
        end
        local i3dNameToExclude = Utils.getNoNil(getXMLString(modDescXMLFile, key .. "#i3dFilename"), "missingI3dFilename")
        if string.find(i3dFilename, i3dNameToExclude) ~= nil then
            self.hasCompressorSound = false
        end
        i = i + 1
    until not self.hasCompressorSound 
    delete(modDescXMLFile)

end

function compressorSound:onPostLoad(savegame)
    
    if self.isClient and self.hasCompressorSound then
        local spec = self.spec_motorized
        local baseString = "sounds"
        local sample = spec.samples
        local xmlSoundFile
        
        if fileExists(_modDirectory .. "sounds/compressorSound.xml") then
            xmlSoundFile = loadXMLFile("compressorSound", _modDirectory .. "sounds/compressorSound.xml")
        else
            print("Error: Script compressorSound.lua: XML-File not found: " .. _modDirectory .. "sounds/compressorSound.xml")
        end
        
        if xmlSoundFile ~= nil then
            local hasAirCompressor = false
            if sample.airCompressorStart ~= nil then
                sample.airCompressorStart.indoorAttributes.volume = 0
                sample.airCompressorStart.outdoorAttributes.volume = 0
                spec.samples.airCompressorStartNew = g_soundManager:loadSampleFromXML(xmlSoundFile, baseString, "airCompressorStartNew", _modDirectory, self.components, 1, AudioGroup.VEHICLE, self.i3dMappings, self)
                spec.samples.airCompressorStartNew.indoorAttributes.volume = spec.samples.airCompressorStartNew.indoorAttributes.volume * _compressorVolumeSetting
                spec.samples.airCompressorStartNew.outdoorAttributes.volume = spec.samples.airCompressorStartNew.outdoorAttributes.volume * _compressorVolumeSetting
                spec.samples.airCompressorStartNew.indoorAttributes.volumeBackup = spec.samples.airCompressorStartNew.indoorAttributes.volume --needed for simpleIC
            end
            if sample.airCompressorRun ~= nil then
                sample.airCompressorRun.indoorAttributes.volume = 0
                sample.airCompressorRun.outdoorAttributes.volume = 0
                spec.samples.airCompressorRunNew = g_soundManager:loadSampleFromXML(xmlSoundFile, baseString, "airCompressorRunNew", _modDirectory, self.components, 0, AudioGroup.VEHICLE, self.i3dMappings, self)
                spec.samples.airCompressorRunNew.indoorAttributes.volume = spec.samples.airCompressorRunNew.indoorAttributes.volume * _compressorVolumeSetting
                spec.samples.airCompressorRunNew.outdoorAttributes.volume = spec.samples.airCompressorRunNew.outdoorAttributes.volume * _compressorVolumeSetting
                spec.samples.airCompressorRunNew.indoorAttributes.volumeBackup = spec.samples.airCompressorRunNew.indoorAttributes.volume --needed for simpleIC
                hasAirCompressor = true
            end
            if sample.airCompressorStop ~= nil then
                sample.airCompressorStop.indoorAttributes.volume = 0
                sample.airCompressorStop.outdoorAttributes.volume = 0
                spec.samples.airCompressorStopNew = g_soundManager:loadSampleFromXML(xmlSoundFile, baseString, "airCompressorStopNew", _modDirectory, self.components, 1, AudioGroup.VEHICLE, self.i3dMappings, self)
                spec.samples.airCompressorStopNew.indoorAttributes.volume = spec.samples.airCompressorStopNew.indoorAttributes.volume * _compressorVolumeSetting
                spec.samples.airCompressorStopNew.outdoorAttributes.volume = spec.samples.airCompressorStopNew.outdoorAttributes.volume * _compressorVolumeSetting
                spec.samples.airCompressorStopNew.indoorAttributes.volumeBackup = spec.samples.airCompressorStopNew.indoorAttributes.volume --needed for simpleIC
            end
            delete(xmlSoundFile)
        end
        
    end

    if self.hasCompressorSound then
        local spec = self.spec_motorized
        local samples = spec.samples    
        if spec.consumersByFillTypeName.air ~= nil then
            local consumer = spec.consumersByFillTypeName.air
            local fillType = self:getFillUnitLastValidFillType(consumer.fillUnitIndex)        
            if fillType == consumer.fillType then
                self.spec_motorized.consumersByFillTypeName.air.usage = self.spec_fillUnit.fillUnits[consumer.fillUnitIndex].capacity * _maxAirUsagePerFullStop  / 340
                if self.isServer then
                    if self:getFillUnitFillLevelPercentage(consumer.fillUnitIndex) > 0.75 then
                        local difference = self.spec_fillUnit.fillUnits[consumer.fillUnitIndex].fillLevel - self.spec_fillUnit.fillUnits[consumer.fillUnitIndex].capacity * 0.75
                        self:addFillUnitFillLevel(self:getOwnerFarmId(), consumer.fillUnitIndex, -difference, consumer.fillType, ToolType.UNDEFINED)
                    end
                end
                self.spec_motorized.consumersByFillTypeName.air.refillLitersPerSecond = self.spec_fillUnit.fillUnits[consumer.fillUnitIndex].capacity / _compressorRunTime * 0.5
                self.spec_motorized.consumersByFillTypeName.air.refillCapacityPercentage = 0.5
            end
        end
    end
    
end

function compressorSound:onStopMotor(noEventSend)
    local spec = self.spec_motorized
    if not spec.isMotorStarted then
        if self.isClient then
            g_soundManager:stopSample(spec.samples.airCompressorStartNew)
            g_soundManager:stopSample(spec.samples.airCompressorStopNew)
            g_soundManager:stopSample(spec.samples.airCompressorRunNew)
        end
    end
end


function compressorSound:onReadUpdateStream(streamId, timestamp, connection)
    if connection.isServer then
        if streamReadBool(streamId) then
            self.streamDoRefill = streamReadBool(streamId)
        end
    end
end
function compressorSound:onWriteUpdateStream(streamId, connection, dirtyMask)
    if not connection.isServer then
        if streamWriteBool(streamId, self.spec_motorized.isMotorStarted) then
            streamWriteBool(streamId, self.streamDoRefill)
        end
    end
end

function compressorSound:onUpdate(dt)
    local spec = self.spec_motorized
    local samples = spec.samples

    if self.attachedAirConsumptionTimer > 0 then
        self.attachedAirConsumptionTimer = self.attachedAirConsumptionTimer - dt  -- 2 seconds release time
        if spec.consumersByFillTypeName.air ~= nil then
            local consumer = spec.consumersByFillTypeName.air
            local fillType = self:getFillUnitLastValidFillType(consumer.fillUnitIndex)
            
            if fillType == consumer.fillType then
                if self.isServer and self.hasCompressorSound then
                    local difference = self.spec_fillUnit.fillUnits[consumer.fillUnitIndex].capacity * (_trailerAttachAirUsagePercent / 100) / 2000 * dt  -- 2 seconds
                    self:addFillUnitFillLevel(self:getOwnerFarmId(), consumer.fillUnitIndex, -difference, consumer.fillType, ToolType.UNDEFINED)
                end
            end
        end
    end

    if spec.isMotorStarted and self.hasCompressorSound then
        if spec.consumersByFillTypeName.air ~= nil then
            local consumer = spec.consumersByFillTypeName.air
            local fillType = self:getFillUnitLastValidFillType(consumer.fillUnitIndex)
            
            if fillType == consumer.fillType then
                local fillLevelPercentage = self:getFillUnitFillLevelPercentage(consumer.fillUnitIndex)
                
                
                local speed = self:getLastSpeed() 
                local difference = 0
                if speed > 22.5 and self.isServer then
                    if speed < spec.motor.maxForwardSpeed * 3.6 - 0.5 then
                        if self.getAxisForward ~= nil then
                            if self:getAxisForward() <= 0 then
                                difference = self.lastMaxSpeed - speed
                            end
                        else
                            difference = self.lastMaxSpeed - speed
                        end
                        if difference > 0 then
                            difference = difference * self.spec_motorized.consumersByFillTypeName.air.usage * 0.05
                            self:addFillUnitFillLevel(self:getOwnerFarmId(), consumer.fillUnitIndex, -difference, consumer.fillType, ToolType.UNDEFINED)
                        end
                        self.lastMaxSpeed = speed
                     else
                        self.lastMaxSpeed = spec.motor.maxForwardSpeed * 3.6 - 0.5
                    end
                else
                    self.lastMaxSpeed = 22.5
                end

                if _debug and self.isClient then
                    if self.getIsEntered ~= nil and self:getIsEntered() then
                        if self.getIsControlled ~= nil and self:getIsControlled() then
                            if not g_gui:getIsGuiVisible() and not g_flightAndNoHUDKeysEnabled then 
                                local x = 0.03
                                local forwardBrake = self.movingDirection > 0 and self:getAxisForward() < 0
                                local brakeIsPressed = self:getLastSpeed() > 1.0 and forwardBrake
                                if brakeIsPressed then
                                    renderText(x, 0.07, 0.035, "Full braking")
                                elseif difference > 0 then
                                    renderText(x, 0.07, 0.035, "Low braking")
                                end                                
                                if g_soundManager:getIsSamplePlaying(samples.airCompressorRunNew) then
                                    renderText(x, 0.12, 0.035, "Air comprossor is running")
                                end
                                fillLevel = tostring(math.floor(fillLevelPercentage * 100 + 0.5)) .. "%"
                                renderText(x, 0.17, 0.035, "Compresed air tank fill level: " .. fillLevel)
                                renderText(x, 0.22, 0.026, 'modDesk:  showIngameDebugInfo="true"')
                            end
                        end
                    end
                end

                if self.isServer then
                    if consumer.doRefill ~= nil then
                        self.streamDoRefill = consumer.doRefill
                    else
                        self.streamDoRefill = false
                    end
                end
                if self.streamDoRefill and self.isClient then
                    if not g_soundManager:getIsSamplePlaying(samples.airCompressorRunNew) then
                        if not g_soundManager:getIsSamplePlaying(samples.motorStart) then
                            g_soundManager:stopSample(samples.airCompressorStopNew)
                            g_soundManager:playSample(samples.airCompressorStartNew)
                            g_soundManager:playSample(samples.airCompressorRunNew, 0, samples.airCompressorStartNew)
                        end
                    end
                else
                    if g_soundManager:getIsSamplePlaying(samples.airCompressorRunNew) then
                        g_soundManager:stopSample(samples.airCompressorRunNew)
                        g_soundManager:playSample(samples.airCompressorStopNew)
                    end
                end
                
            end
        end
    end
end
