--[[
DayTimeRunningLights

Specialization for day time running lights on an vehicle

Author:		Ifko[nator]
Datum:		11.08.2019
Version:	v1.0

History:	v1.0 @ 11.08.2019 - initial implementation in FS 19
			-------------------------------------------------------------------------------------------------------------------------------------------------------------
]]

DayTimeRunningLights = {};
DayTimeRunningLights.debugPriority = 0;

local function printError(errorMessage, isWarning, isInfo)
	local prefix = "::ERROR:: ";
	
	if isWarning then
		prefix = "::WARNING:: ";
	elseif isInfo then
		prefix = "::INFO:: ";
	end;
	
	print(prefix .. "from the DayTimeRunningLights.lua: " .. tostring(errorMessage));
end;

local function printDebug(debugMessage, priority, addString)
	if DayTimeRunningLights.debugPriority >= priority then
		local prefix = "";
		
		if addString then
			prefix = "::DEBUG:: from the DayTimeRunningLights.lua: ";
		end;
		
		setFileLogPrefixTimestamp(false);
		print(prefix .. tostring(debugMessage));
		setFileLogPrefixTimestamp(true);
	end;
end;

function DayTimeRunningLights.prerequisitesPresent(specializations)
    return SpecializationUtil.hasSpecialization(Lights, specializations); 
end;

function DayTimeRunningLights.registerEventListeners(vehicleType)
	local functionNames = {
        "onPreLoad",
		"onLoad",
		"onUpdate"
	};

	for _, functionName in ipairs(functionNames) do
		SpecializationUtil.registerEventListener(vehicleType, functionName, DayTimeRunningLights);
	end;
end;

function DayTimeRunningLights:onPreLoad(savegame)
    self.setLightsTypesMask = Utils.prependedFunction(self.setLightsTypesMask, DayTimeRunningLights.setLightsTypesMask);
end;

function DayTimeRunningLights:onLoad(savegame)
    DayTimeRunningLights.debugPriority = Utils.getNoNil(getXMLInt(self.xmlFile, "vehicle.lights.daytimeRunningLights#debugPriority"), DayTimeRunningLights.debugPriority);
    
    local specLights = self.spec_lights;

    specLights.daytimeRunningLights = {};
    specLights.lightsAreOn = false;

    printDebug("", 1, false);
	printDebug("-----------------------------------------------Debug from the DayTimeRunningLights.lua Start------------------------------------------------", 1, false);

    local daytimeRunningLightNumber = 0;
        
    while true do
        local daytimeRunningLightKey = "vehicle.lights.daytimeRunningLights.daytimeRunningLight(" .. tostring(daytimeRunningLightNumber) .. ")";

        if not hasXMLProperty(self.xmlFile, daytimeRunningLightKey) then
            printDebug("", 1, false);
            printDebug("-----------------------------------------------Debug from the DayTimeRunningLights.lua end--------------------------------------------------", 1, false);
        
            break;
        end;

        local daytimeRunningLight = {};
       
        daytimeRunningLight.node = I3DUtil.indexToObject(self.components, getXMLString(self.xmlFile, daytimeRunningLightKey .. "#node"), self.i3dMappings);
        
        if daytimeRunningLight.node ~= nil then
            daytimeRunningLight.intensity = Utils.getNoNil(getXMLInt(self.xmlFile, daytimeRunningLightKey .. "#intensity"), 25);
				
		    setVisibility(daytimeRunningLight.node, false);		
            printDebug("Load day time running light number '" .. daytimeRunningLightNumber + 1 .. "' successfully.", 1, true);
            
            table.insert(specLights.daytimeRunningLights, daytimeRunningLight);
        else
            printError("Missing the 'node' for the day time running light number '" .. daytimeRunningLightNumber + 1 .. "'! This day time running light will be ignored!", false, false);
        end;
    
        daytimeRunningLightNumber = daytimeRunningLightNumber + 1;
    end;
end;

function DayTimeRunningLights:onUpdate(dt, isActiveForInput, isActiveForInputIgnoreSelection, isSelected)
    if self:getIsActive() then
        local specLights = self.spec_lights;
        
        for _, daytimeRunningLight in pairs(specLights.daytimeRunningLights) do
            local _, y, z, w = getShaderParameter(daytimeRunningLight.node, "lightControl");

            setShaderParameter(daytimeRunningLight.node, "lightControl", math.max(1 * daytimeRunningLight.intensity, 0), y, z, w, false);
            setVisibility(daytimeRunningLight.node, self:getIsMotorStarted() and not specLights.lightsAreOn);
        end;
    end;
end;

function DayTimeRunningLights:setLightsTypesMask(lightsTypesMask, force, noEventSend)
    local specLights = self.spec_lights;

    specLights.lightsAreOn = bitAND(lightsTypesMask, 1) ~= 0;
end;