--[[
BuyableSteeringAxle

Script to buy steering axle for trailers with an configuration

@Author:	Ifko[nator]
@Date:		08.03.2018

@Version:	V1.5

@History:	V 1.0 @06.12.2017 - initial release in FS 17
			V 1.5 @08.03.2018 - support for steering axles with an steering node (Thanks to Crouwler who tell me about this!)
]]

BuyableSteeringAxle = {};

function BuyableSteeringAxle.prerequisitesPresent(specializations)
    return SpecializationUtil.hasSpecialization(Trailer, specializations);
end;

function BuyableSteeringAxle:load(savegame)
	self.steeringAxle = {};
	
	self.steeringAxle.isBought = false;
	
	if #self.wheels == 0 then
		return;
	end;
	
	self.steeringAxle.wheels = {};
    
    local steeringAxleNumber = 0;
    
    while true do
        local steeringAxleKey = "vehicle.buyableSteeringAxles.buyableSteeringAxle(" .. tostring(steeringAxleNumber) .. ")";
        
        if not hasXMLProperty(self.xmlFile, steeringAxleKey) then
            break;
        end;
        
        local steeringWheel = {};
        
        local repr = Utils.indexToObject(self.components, getXMLString(self.xmlFile, steeringAxleKey .. "#wheelRepr"));
        local wheelNumber = getXMLInt(self.xmlFile, steeringAxleKey .. "#wheelNumber");
		
        if repr ~= nil or wheelNumber ~= nil then
			if repr ~= nil then
				steeringWheel.repr = repr;
			elseif wheelNumber ~= nil then
				steeringWheel.wheelNumber = wheelNumber;
			end;
			
			steeringWheel.configName = getXMLString(self.xmlFile, steeringAxleKey .. "#configName");
			steeringWheel.activeConfig = getXMLString(self.xmlFile, steeringAxleKey .. "#activeConfig");
        	
            if steeringWheel.configName ~= nil and steeringWheel.configName ~= "" and steeringWheel.activeConfig ~= nil and steeringWheel.activeConfig ~= "" then
				steeringWheel.steeringAxleScale = Utils.getNoNil(getXMLFloat(self.xmlFile, steeringAxleKey .. "#steeringAxleScale"), 0.3);
                steeringWheel.steeringAxleRotMin = Utils.degToRad(Utils.getNoNil(getXMLFloat(self.xmlFile, steeringAxleKey .. "#steeringAxleRotMin"), -15));
                steeringWheel.steeringAxleRotMax = Utils.degToRad(Utils.getNoNil(getXMLFloat(self.xmlFile, steeringAxleKey .. "#steeringAxleRotMax"), 15));
				
				local steeringNode = Utils.indexToObject(self.components, getXMLString(self.xmlFile, steeringAxleKey .. "#steeringNode"));
				
				if steeringNode ~= nil then
					steeringWheel.steeringNode = steeringNode;
					steeringWheel.steeringNodeMinTransX = Utils.getNoNil(getXMLFloat(self.xmlFile, steeringAxleKey .. "#steeringNodeMinTransX"), 0.116);
					steeringWheel.steeringNodeMaxTransX = Utils.getNoNil(getXMLFloat(self.xmlFile, steeringAxleKey .. "#steeringNodeMaxTransX"), -0.116);
				end;
				
                table.insert(self.steeringAxle.wheels, steeringWheel);
            end;
        end;
        
        steeringAxleNumber = steeringAxleNumber + 1;
    end;
	
	if #self.steeringAxle.wheels > 0 then
        local storeItem = StoreItemsUtil.storeItemsByXMLFilename[self.configFileName:lower()];
        
        for _, steeringWheel in pairs(self.steeringAxle.wheels) do
            if storeItem.configurations ~= nil and storeItem.configurations[steeringWheel.configName] ~= nil then
				local configurations = storeItem.configurations[steeringWheel.configName];
                local config = configurations[self.configurations[steeringWheel.configName]];
                
                for wheelNumber, wheel in pairs(self.wheels) do
                    if (steeringWheel.repr ~= nil and wheel.repr == steeringWheel.repr) or wheelNumber == steeringWheel.wheelNumber then	
                        if g_i18n:hasText(steeringWheel.activeConfig) then
							steeringWheel.activeConfig = g_i18n:getText(steeringWheel.activeConfig);
						end;
						
						if config.name == steeringWheel.activeConfig then
                            wheel.steeringAxleScale = steeringWheel.steeringAxleScale;
                            wheel.steeringAxleRotMin = steeringWheel.steeringAxleRotMin;
                            wheel.steeringAxleRotMax = steeringWheel.steeringAxleRotMax;
							
							if steeringWheel.steeringNode ~= nil then
								wheel.steeringNode = steeringWheel.steeringNode;
								wheel.steeringNodeMinTransX = steeringWheel.steeringNodeMinTransX;
								wheel.steeringNodeMaxTransX = steeringWheel.steeringNodeMaxTransX;
								
								if wheel.steeringNodeMaxRot == 0 then	
									wheel.steeringNodeMaxRot = wheel.steeringAxleRotMax;
								end;
								
								if wheel.steeringNodeMinRot == -0 then
									wheel.steeringNodeMinRot = wheel.steeringAxleRotMin;
								end;
							end;
							
							self.steeringAxle.isBought = true;
							self.steeringAxleUpdateBackwards = true;
                        else
                            wheel.steeringAxleScale = 0;
                            wheel.steeringAxleRotMin = 0;
                            wheel.steeringAxleRotMax = 0;
							wheel.steeringNodeMaxRot = 0;
							wheel.steeringNodeMinRot = 0;
							
							if steeringWheel.steeringNode ~= nil then
								wheel.steeringNodeMinTransX = 0;
								wheel.steeringNodeMaxTransX = 0;
							end;
                        end;
                    end;
                end;
            end;
        end;
    end;
	
	self.setSteeringAxleActive = BuyableSteeringAxle.setSteeringAxleActive;
	
	self.lockSteeringAxle = false;
	self.steerAxleIsLocked = false;
	
	if savegame ~= nil and not savegame.resetVehicles then
		self.lockSteeringAxle = Utils.getNoNil(getXMLBool(savegame.xmlFile, savegame.key .. "#lockSteeringAxle"), false);
	end;
	
	self:setSteeringAxleActive(self.lockSteeringAxle, false);
end;

function BuyableSteeringAxle:writeStream(streamId, connection)
	if self.setSteeringAxleActive ~= nil then	
		streamWriteBool(streamId, self.steerAxleIsLocked);
	end;
end;

function BuyableSteeringAxle:readStream(streamId, connection)
	if self.setSteeringAxleActive ~= nil then	
		local steerAxleIsLocked = streamReadBool(streamId);

		self:setSteeringAxleActive(steerAxleIsLocked, true);
	end;
end;

function BuyableSteeringAxle:update(dt)
	if self.isClient and self.steeringAxle.isBought then
		if self:getIsActiveForInput() then
			if InputBinding.hasEvent(InputBinding.TOGGLE_CHOPPER) then
				self:setSteeringAxleActive(not self.lockSteeringAxle, false);
			end;
		end;
		
		self.updateSteeringAxleAngle = not self.lockSteeringAxle;
		
		if self.lockSteeringAxle then
			if self.steeringAxleAngle > 0 then
				self.steeringAxleAngle = math.max(self.steeringAxleAngle - self.steeringAxleAngleSpeed * dt, 0);
			else
				self.steeringAxleAngle = math.min(self.steeringAxleAngle + self.steeringAxleAngleSpeed * dt, 0);
			end;
		end;
	end;
end;

function BuyableSteeringAxle:setSteeringAxleActive(lockSteeringAxle, noEventSend)
	if lockSteeringAxle ~= self.lockSteeringAxle then
		self.lockSteeringAxle = lockSteeringAxle;
		
		BuyableSteeringAxleEvent.sendEvent(self, lockSteeringAxle, noEventSend);
	end;
end;

function BuyableSteeringAxle:draw()
	if self.isClient and self:getIsActiveForInput() and self.steeringAxle.isBought then
		if self.lockSteeringAxle then
			g_currentMission:addHelpButtonText(g_i18n:getText("UNLOCK_STEERING_AXLE"), InputBinding.TOGGLE_CHOPPER);
		else
			g_currentMission:addHelpButtonText(g_i18n:getText("LOCK_STEERING_AXLE"), InputBinding.TOGGLE_CHOPPER);
		end;
	end;
end;

function BuyableSteeringAxle:getSaveAttributesAndNodes(nodeIdent)
	if self.steeringAxleIsBought then
		return 'lockSteeringAxle="' .. tostring(self.lockSteeringAxle) .. '"', nil;
	else
		return nil, nil;
	end;
end;

--## unused but needed functions

function BuyableSteeringAxle:delete()end;
function BuyableSteeringAxle:mouseEvent(posX, posY, isDown, isUp, button)end;
function BuyableSteeringAxle:keyEvent(unicode, sym, modifier, isDown)end;
function BuyableSteeringAxle:updateTick(dt)end;

--## MP Stuff

--## turn steering axle on/off

BuyableSteeringAxleEvent = {};
BuyableSteeringAxleEvent_mt = Class(BuyableSteeringAxleEvent, Event);

InitEventClass(BuyableSteeringAxleEvent, "BuyableSteeringAxleEvent");

function BuyableSteeringAxleEvent:emptyNew()
	local self = Event:new(BuyableSteeringAxleEvent_mt);
    
	return self;
end;

function BuyableSteeringAxleEvent:new(trailer, lockSteeringAxle)
	local self = BuyableSteeringAxleEvent:emptyNew();
	
	self.trailer = trailer;
	self.steerAxleIsLocked = lockSteeringAxle;
	
	return self;
end;

function BuyableSteeringAxleEvent:readStream(streamId, connection)
	local id = streamReadInt32(streamId);
	
	self.trailer = networkGetObject(id);
	self.steerAxleIsLocked = streamReadBool(streamId);
    self:run(connection);
end;

function BuyableSteeringAxleEvent:writeStream(streamId, connection)
	streamWriteInt32(streamId, networkGetObjectId(self.trailer));
	streamWriteBool(streamId, self.steerAxleIsLocked);
end;

function BuyableSteeringAxleEvent:run(connection)
	self.trailer:setSteeringAxleActive(self.steerAxleIsLocked, true);
	
	if not connection:getIsServer() then
		g_server:broadcastEvent(BuyableSteeringAxleEvent:new(self.trailer, self.steerAxleIsLocked), nil, connection, self.trailer);
	end;
end;

function BuyableSteeringAxleEvent.sendEvent(trailer, lockSteeringAxle, noEventSend)
	if noEventSend == nil or noEventSend == false then
		if g_server ~= nil then
			g_server:broadcastEvent(BuyableSteeringAxleEvent:new(trailer, lockSteeringAxle), nil, nil, trailer);
		else
			g_client:getServerConnection():sendEvent(BuyableSteeringAxleEvent:new(trailer, lockSteeringAxle));
		end;
	end;
end;