--[[
AutomaticHeadlights

Script for configurable automatic headlights

@Author:	Ifko[nator]
@Date:		26.02.2018

@Version:	1.0 - initial release in FS 17
]]

AutomaticHeadlights = {};

AutomaticHeadlights.debugPriority = 0;

local function printError(errorMessage, isWarning, isInfo)
	local prefix = "::ERROR:: ";
	
	if isWarning then
		prefix = "::WARNING:: ";
	elseif isInfo then
		prefix = "::INFO:: ";
	end;
	
	print(prefix .. "from the AutomaticHeadlights.lua: " .. tostring(errorMessage));
end;

local function printDebug(debugMessage, priority, addString)
	if AutomaticHeadlights.debugPriority >= priority then
		local prefix = "";
		
		if addString then
			prefix = "::DEBUG:: from the AutomaticHeadlights.lua: ";
		end;
		
		print(prefix .. tostring(debugMessage));
	end;
end;

function AutomaticHeadlights.prerequisitesPresent(specializations)
    return SpecializationUtil.hasSpecialization((Motorized and Lights), specializations); 
end;

function AutomaticHeadlights:preLoad(savegame)
    self.setLightsTypesMask = Utils.prependedFunction(self.setLightsTypesMask, AutomaticHeadlights.setLightsTypesMask);
end;

function AutomaticHeadlights:load(savegame)	
    AutomaticHeadlights.debugPriority = Utils.getNoNil(getXMLFloat(self.xmlFile, "vehicle.automaticHeadlights#debugPriority"), AutomaticHeadlights.debugPriority);
	
	self.automaticHeadlight = {};
    
    --## make this configurable
    self.automaticHeadlight.isBought = false;
    self.automaticHeadlight.configName = getXMLString(self.xmlFile, "vehicle.automaticHeadlights#configName");
    self.automaticHeadlight.activeConfig = getXMLString(self.xmlFile, "vehicle.automaticHeadlights#activeConfig");
	
	printDebug("", 1, false);
	printDebug("-----------------------------------------------Debug from the AutomaticHeadlights.lua Start------------------------------------------------", 1, false);
	
    if self.automaticHeadlight.configName ~= nil 
        and self.automaticHeadlight.configName ~= ""
        and self.automaticHeadlight.activeConfig ~= nil
        and self.automaticHeadlight.activeConfig ~= ""
    then
        local storeItem = StoreItemsUtil.storeItemsByXMLFilename[self.configFileName:lower()];
        
        if storeItem.configurations ~= nil and storeItem.configurations[self.automaticHeadlight.configName] ~= nil then
            local configurations = storeItem.configurations[self.automaticHeadlight.configName];
            local config = configurations[self.configurations[self.automaticHeadlight.configName]];
        	local configName = self.automaticHeadlight.activeConfig;
			
			if g_i18n:hasText(self.automaticHeadlight.activeConfig) then
				configName = g_i18n:getText(self.automaticHeadlight.activeConfig);
			end;
			
			printDebug("active config name for automatic head light = " .. configName, 1, true);
			
            self.automaticHeadlight.isBought = config.name == configName;
		else
			printError("Missing the configuration '" .. self.automaticHeadlight.configName .. "'! The script will stop for the vehicle '" .. self.configFileName .. "' now!", false, false);
        end;
	else
		if self.automaticHeadlight.configName == nil or self.automaticHeadlight.configName == "" then
			printError("Missing the 'configName' attribute for the automatic headlight configuration! The script will stop for the vehicle '" .. self.configFileName .. "' now!", false, false);
		elseif self.automaticHeadlight.activeConfig == nil or self.automaticHeadlight.activeConfig == "" then
			printError("Missing the 'activeConfig' attribute for the automatic headlight configuration! The script will stop for the vehicle '" .. self.configFileName .. "' now!", false, false);
		end;
    end;
    
    if self.automaticHeadlight.isBought then
        printDebug("", 1, false);
		printDebug("Automatic headlight is active in '" .. self.configFileName .. "'.", 1, true, false);
		printDebug("", 1, false);
		printDebug("-----------------------------------------------Debug from the automatic headlight start------------------------------------------------", 1, false);
		printDebug("", 1, false);
		
		self.automaticHeadlights = {};
        
        local automaticHeadlightNumber = 0;
        
        while true do
            local automaticHeadlightKey = "vehicle.automaticHeadlights.automaticHeadlight(" .. tostring(automaticHeadlightNumber) .. ")";
            
            if not hasXMLProperty(self.xmlFile, automaticHeadlightKey) then
				printDebug("", 1, false);
				printDebug("-----------------------------------------------Debug from the automatic headlight end--------------------------------------------------", 1, false);
			
                break;
            end;
            
            local automaticHeadlightCoronaIndex = Utils.indexToObject(self.components, getXMLString(self.xmlFile, automaticHeadlightKey .. "#decoration"));
            local automaticHeadlightRealLightIndex = Utils.indexToObject(self.components, getXMLString(self.xmlFile, automaticHeadlightKey .. "#realLight"));
            
            if automaticHeadlightCoronaIndex ~= nil then
                local automaticHeadlight = {};
                
                automaticHeadlight.coronaNode = automaticHeadlightCoronaIndex;
                
                local scaleX, scaleY, scaleZ = getScale(automaticHeadlight.coronaNode);
                
                automaticHeadlight.coronaStartScaleX = scaleX;
                automaticHeadlight.coronaStartScaleY = scaleY;
                automaticHeadlight.coronaStartScaleZ = scaleZ;
				
				printDebug("Load automatic headlight (corona) number " .. automaticHeadlightNumber + 1 .. " successfully.", 1, true);
                
                if automaticHeadlightRealLightIndex ~= nil then
                    printDebug("Load automatic headlight (real light) number " .. automaticHeadlightNumber + 1 .. " successfully.", 1, true);
					
					automaticHeadlight.realNode = automaticHeadlightRealLightIndex;
                
                    automaticHeadlight.startLightRange = getLightRange(automaticHeadlight.realNode);
				else
					printDebug("Missing the 'realLightNode' attribute for the automatic headlight number '" .. automaticHeadlightNumber + 1 .. "'! This automatic headlight will run without an real light.", 2, true);
                end;
                
                table.insert(self.automaticHeadlights, automaticHeadlight);
			else
				printError("Missing the 'coronaNode' attribute for the automatic headlight number '" .. automaticHeadlightNumber + 1 .. "'! The script will skip this automatic headlight now!", false, false);
            end;
        
            automaticHeadlightNumber = automaticHeadlightNumber + 1;
        end;
        
		self.getIsRainTypeActive = AutomaticHeadlights.getIsRainTypeActive;
        self.getIsFoggy = AutomaticHeadlights.getIsFoggy;
        self.getShouldTurnOnLights = AutomaticHeadlights.getShouldTurnOnLights;
		
        self.automaticHeadlight.turnLightOn = true;
		self.automaticHeadlight.turnLightOff = false;
        
        self.automaticHeadlight.turnOffTimer = 0;
        self.automaticHeadlight.lightDelay = Utils.getNoNil(getXMLString(self.xmlFile, "vehicle.automaticHeadlights#lightDelay"), 15) * 1000; --## default 15 sec
        self.automaticHeadlight.turnOnTime = Utils.getNoNil(getXMLFloat(self.xmlFile, "vehicle.automaticHeadlights#turnOnTime"), 20); --## default 20
        self.automaticHeadlight.turnOffTime = Utils.getNoNil(getXMLFloat(self.xmlFile, "vehicle.automaticHeadlights#turnOffTime"), 7) - 1; --## default 6
        self.automaticHeadlight.lightsAreOn = false;
	else
		printDebug("", 1, false);
		printDebug("Automatic headlight is not active in '" .. self.configFileName .. "'.", 1, true);
		printDebug("", 1, false);
		printDebug("-----------------------------------------------Debug from the automatic headlight end--------------------------------------------------", 1, false);
    end;
end;

function AutomaticHeadlights:update(dt)
    if self.automaticHeadlight.isBought then
        if self.automaticHeadlight.lightsAreOn and not self.isMotorStarted and self:getIsActive() then
            self.automaticHeadlight.turnOffTimer = self.automaticHeadlight.lightDelay + g_currentMission.time;
        end;
		
		if AutomaticHeadlights.debugPriority >= 1 and self:getIsActive() then
			renderText(0.5, 0.5, 0.02, "self.automaticHeadlight.turnOffTimer = " .. self.automaticHeadlight.turnOffTimer);
			renderText(0.5, 0.48, 0.02, "self.automaticHeadlight.lightDelay = " .. self.automaticHeadlight.lightDelay);
			renderText(0.5, 0.46, 0.02, "self.automaticHeadlight.lightsAreOn = " .. tostring(self.automaticHeadlight.lightsAreOn));
			renderText(0.5, 0.44, 0.02, "self.automaticHeadlight.turnLightOn = " .. tostring(self.automaticHeadlight.turnLightOn));
			renderText(0.5, 0.42, 0.02, "self.automaticHeadlight.turnLightOff = " .. tostring(self.automaticHeadlight.turnLightOff));
			renderText(0.5, 0.4, 0.02, "self:getIsActive() = " .. tostring(self:getIsActive()));
			renderText(0.5, 0.38, 0.02, "self.isMotorStarted = " .. tostring(self.isMotorStarted));
		end;
        
        if self.automaticHeadlights ~= nil then
            for _, automaticHeadlight in pairs(self.automaticHeadlights) do
                if self.automaticHeadlight.turnOffTimer ~= 0 
					and self.automaticHeadlight.turnOffTimer < g_currentMission.time 
					and not self.isMotorStarted 
					and not self:getIsActive() 
					and self.automaticHeadlight.lightsAreOn 
				then
                    if automaticHeadlight.coronaNode ~= nil then	
                        local scaleX, scaleY, scaleZ = getScale(automaticHeadlight.coronaNode);
                        
                        scaleX = math.max(scaleX - (0.005 * automaticHeadlight.coronaStartScaleX), 0);
                        scaleY = math.max(scaleY - (0.005 * automaticHeadlight.coronaStartScaleY), 0);
                        scaleZ = math.max(scaleZ - (0.005 * automaticHeadlight.coronaStartScaleZ), 0);
                        
                        setScale(automaticHeadlight.coronaNode, scaleX, scaleY, scaleZ);
                        
                        if scaleX and scaleY and scaleZ == 0 then
							self:setLightsTypesMask(0, true, true);
                            self.automaticHeadlight.turnOffTimer = 0;
                        	
                            self.automaticHeadlight.turnLightOn = true;
							
							printDebug("Turn lights off (cororna).", 1, true);
                        end;
                    end;
                    
                    if automaticHeadlight.realNode ~= nil then
                        local lightRange = getLightRange(automaticHeadlight.realNode);
                        
                        lightRange = math.max(lightRange - (lightRange * 0.015), 0);
                        
                        setLightRange(automaticHeadlight.realNode, lightRange);
                        
                        if lightRange == 0 then
                            --## this should be not needed, but save is save ..
                            self:setLightsTypesMask(0, true, true);
                            self.automaticHeadlight.turnOffTimer = 0;
							
                            self.automaticHeadlight.turnLightOn = true;
							
							printDebug("Turn lights off (realLight).", 1, true);
                        end;
                    end;
                else
                    if automaticHeadlight.coronaNode ~= nil then	
                        setScale(automaticHeadlight.coronaNode, automaticHeadlight.coronaStartScaleX, automaticHeadlight.coronaStartScaleY, automaticHeadlight.coronaStartScaleZ);
                    end;
                    
                    if automaticHeadlight.realNode ~= nil then				
                        setLightRange(automaticHeadlight.realNode, automaticHeadlight.startLightRange);
                    end;
                end;
            end;
        end;
        
		local turnLightOn = self:getShouldTurnOnLights() or self:getIsFoggy() or self:getIsRainTypeActive();

		printDebug("Should turn on lights = " .. tostring(turnLightOn) .. ".", 3, true);
		
		if turnLightOn then
			if not self.automaticHeadlight.lightsAreOn then
				if self.isMotorStarted then
					if turnLightOn and self.automaticHeadlight.turnLightOn then
						if AutomaticHeadlights.debugPriority >= 1 then
							number = Utils.getNoNil(number, 0) + 1; printDebug("Change light state number: " .. number .. ".", 1, true);
						end;
						
						Steerable.setNextLightsState(self);
						
						self.automaticHeadlight.turnOffTimer = 0;
						
						self.automaticHeadlight.turnLightOn = false;
						self.automaticHeadlight.turnLightOff = true;
					end;
				else
					self.automaticHeadlight.turnLightOn = true;
				end;
			end;
		else
			if self.automaticHeadlight.turnLightOff then
                self:setLightsTypesMask(0, true, true);
				
				self.automaticHeadlight.turnOffTimer = 0;
				
				self.automaticHeadlight.turnLightOff = false;
				self.automaticHeadlight.turnLightOn = true;
			end;
		end;
    end;
end;

function AutomaticHeadlights:setLightsTypesMask(lightsTypesMask, force, noEventSend)
    self.automaticHeadlight.lightsAreOn = bitAND(lightsTypesMask, 1) ~= 0;
end;

function AutomaticHeadlights:getIsFoggy()
    return g_currentMission.environment.currentRain ~= nil and g_currentMission.environment.currentRain.rainTypeId == Environment.RAINTYPE_FOG;
end;

function AutomaticHeadlights:getIsRainTypeActive()
	if g_currentMission.environment.currentRain ~= nil
        and g_currentMission.environment.currentRain.rainTypeId ~= Environment.RAINTYPE_FOG
        and g_currentMission.environment.currentRain.rainTypeId ~= Environment.RAINTYPE_CLOUDY
    then
        local rainType = g_currentMission.environment:getRainType();
        local isRainTypeActive = false;
        
        if rainType ~= nil then
            if rainType.rootNode ~= nil then
                isRainTypeActive = getVisibility(rainType.rootNode);
            end;
        end;
	
		return isRainTypeActive;
	end;
	
	return false;
end;

function AutomaticHeadlights:getShouldTurnOnLights()
    return g_currentMission.environment.currentHour >= self.automaticHeadlight.turnOnTime or g_currentMission.environment.currentHour <= self.automaticHeadlight.turnOffTime;
end;

--## unused but needed functions

function AutomaticHeadlights:delete()end;
function AutomaticHeadlights:mouseEvent(posX, posY, isDown, isUp, button)end;
function AutomaticHeadlights:keyEvent(unicode, sym, modifier, isDown)end;
function AutomaticHeadlights:draw()end;