--[[
	en: 'fill weight' Specialization for Farming Simulator 17
	de: 'Füllgewicht' Spezialisierung für den Landwirtschafts Simulator 17
	
	en: this script adds a scale to all trailer's/shovel's/forage mixing wagon's/sowing machine's/combine's to show you the ingame current weight of the load.
	de: Dieses Script fügt allen Anhängern/Schaufeln/Futtermischwagen/Sämaschinen/Dreschern eine Waage hinzu, die das aktuelle Ingamegewicht des Ladeguts anzeigt.
	
	Author: 	Ifko[nator]
	Date:		05.12.2017
	Version:	2.5
	
	History:
				v1.0 @ 22.04.2016 - intial implementation in FS 15
				v2.0 @ 26.10.2016 - intial implementation in FS 17
				v2.1 @ 23.05.2017 - removed hud and placed the render function to the vehicle hud
				v2.5 @ 05.12.2017 - added support for the straw harvest addon by creative mesh (Krone Bale Collect..)
]]

FillWeight = {};

FillWeight.modDirectory = g_currentModDirectory;

function FillWeight.prerequisitesPresent(specializations)
	return SpecializationUtil.hasSpecialization(Fillable, specializations);
end;

function FillWeight:load(savegame)
	self.fillWeight = {};
	
	self.getBaleWeight = FillWeight.getBaleWeight;
	self.allowRenderText = FillWeight.allowRenderText;
	
	local textWidth = g_currentMission.vehicleHudBg.height * 0.15 / g_screenAspectRatio;
	
	self.fillWeight.textPosX = g_currentMission.vehicleHudBg.x + textWidth * 11.35;
	self.fillWeight.textPosY = g_currentMission.cruiseControlOverlay.y + (g_currentMission.cruiseControlTextOffsetY * 6.85);
	
	self.fillWeight.savedBaleWeight = 0;
	self.fillWeight.lastFillLevel = self:getFillLevel();
	
	if savegame ~= nil then
		self.fillWeight.savedBaleWeight = Utils.getNoNil(getXMLFloat(savegame.xmlFile, savegame.key .. "#savedBaleWeight"), self.fillWeight.savedBaleWeight);
	end;
	
	self.fillWeight.allowedAddWeight = true;
	self.fillWeight.disableVehicle = Utils.getNoNil(getXMLBool(self.xmlFile, "vehicle.fillWeight#disableVehicle"), false);
	
	local modDesc = loadXMLFile("modDesc", FillWeight.modDirectory .. "modDesc.xml");
	local vehicleNumber = 0;
	local disableVehicle = false;
	
	while true do
		local getKey = "modDesc.disableFillWeights.disableFillWeight(" .. tostring(vehicleNumber) .. ")";
		
		if not hasXMLProperty(modDesc, getKey) then
			break;
		end;
		
		local i3dName = Utils.getNoNil(getXMLString(modDesc, getKey .. "#i3dName"), "No i3dName given!");
		local i3dFilename = getXMLString(self.xmlFile, "vehicle.filename");
		
		if i3dName == "" then
			i3dName = "No i3dName given!";
		end;
	
		if string.find(i3dFilename, i3dName) then	
			disableVehicle = true;
		end;
		
		vehicleNumber = vehicleNumber + 1;
	end;
	
	delete(modDesc);
	
	if not self.fillWeight.disableVehicle then
		self.fillWeight.disableVehicle = disableVehicle;
	end;
	
	self.isClassicsPackBaleLoader = self.freeBaleSliders ~= nil;
end;

function FillWeight:draw()
	local fillWeightPerLiter = 0;
	
	for i, fillUnit in pairs(self.fillUnits) do
		if fillUnit.currentFillType ~= nil 
			and fillUnit.currentFillType ~= FillUtil.FILLTYPE_UNKNOWN 
			and FillUtil.fillTypeIndexToDesc[fillUnit.currentFillType].massPerLiter ~= nil 
			and not (
				self.isClassicsPackBaleLoader 
				or fillUnit.currentFillType == FillUtil.FILLTYPE_SQUAREBALE
				or fillUnit.currentFillType == FillUtil.FILLTYPE_ROUNDBALE
				or fillUnit.currentFillType == FillUtil.FILLTYPE_PIG
				or fillUnit.currentFillType == FillUtil.FILLTYPE_SHEEP
				or fillUnit.currentFillType == FillUtil.FILLTYPE_COW
			) 
		then
			fillWeightPerLiter = FillUtil.fillTypeIndexToDesc[fillUnit.currentFillType].massPerLiter * 1000;
		elseif fillUnit.currentFillType == FillUtil.FILLTYPE_SQUAREBALE or fillUnit.currentFillType == FillUtil.FILLTYPE_ROUNDBALE then
			if self.baleGrabber ~= nil then
				self.fillWeight.savedBaleWeight = self.fillWeight.savedBaleWeight + self:getBaleWeight(fillUnit.currentFillType == FillUtil.FILLTYPE_ROUNDBALE);
			else
				if (self:getFillLevel() == 1 and self.fillWeight.savedBaleWeight == 0) or self.fillWeight.lastFillLevel ~= self:getFillLevel() then	
					if fillUnit.currentFillType == FillUtil.FILLTYPE_ROUNDBALE then
						self.fillWeight.savedBaleWeight = self.fillWeight.savedBaleWeight + math.random(260, 320);
					else
						self.fillWeight.savedBaleWeight = self.fillWeight.savedBaleWeight + math.random(450, 510);
					end;
					
					self.fillWeight.lastFillLevel = self:getFillLevel();
				end;
			end;
		elseif self.isClassicsPackBaleLoader then
			if (self:getFillLevel() == 1 and self.fillWeight.savedBaleWeight == 0) or self.fillWeight.lastFillLevel ~= self:getFillLevel() then	
				self.fillWeight.savedBaleWeight = self.fillWeight.savedBaleWeight + math.random(15, 20);
			
				self.fillWeight.lastFillLevel = self:getFillLevel();
			end;
		elseif fillUnit.currentFillType == FillUtil.FILLTYPE_PIG then
			fillWeightPerLiter = 350;
		elseif fillUnit.currentFillType == FillUtil.FILLTYPE_SHEEP then
			fillWeightPerLiter = 160;
		elseif fillUnit.currentFillType == FillUtil.FILLTYPE_COW then
			fillWeightPerLiter = 1100;
		end;
	end;
	
	local fillWeight = self:getFillLevel() * fillWeightPerLiter;
	
	if self.fillWeight.savedBaleWeight > 0 and self:getFillLevel() > 0 then
		fillWeight = self.fillWeight.savedBaleWeight;
	else
		self.fillWeight.savedBaleWeight = 0;
	end;
	
	if self:allowRenderText() then
		setTextAlignment(RenderText.ALIGN_RIGHT);
		setTextBold(true);
		setTextColor(0.15, 0.15, 0.15, 1);
		
		local precision = 0; --## how many decimal places you want?!
		
		renderText(self.fillWeight.textPosX, self.fillWeight.textPosY, g_currentMission.cruiseControlTextSize, string.format("%s kg", g_i18n:formatNumber(fillWeight, precision)));
	end;
	
	--## restore the render texts values to standard by leaving the function, to avoid conflicts with render text values from other scripts
	setTextAlignment(RenderText.ALIGN_LEFT);
	setTextBold(false);
	setTextColor(1, 1, 1, 1);
end;

function FillWeight:getBaleWeight(isRoundbale)
	local baleVolume, baleWeight, newWeight = 0, 0, 0;
	
	if self.baleGrabber.currentBale ~= nil then	
		local bale = networkGetObject(self.baleGrabber.currentBale);
		
		if bale ~= nil then
			local baleFillType = Utils.getNoNil(FillUtil.fillTypeIntToName[bale.fillType], "straw");
			
			if self.fillWeight.allowedAddWeight then
				if isRoundbale then
					local baleWidth = Utils.round(Utils.getNoNil(tonumber(bale.baleWidth), 1.12), 2);
					local baleDiameter = Utils.round(Utils.getNoNil(tonumber(bale.baleDiameter), 1.3), 2);
					
					--print("current Bale: fillType = " .. baleFillType .. " bale width = " .. baleWidth .. " bale diameter = " .. baleDiameter .. " isRoundbale = " .. tostring(isRoundbale));
					
					baleVolume = baleWidth * baleDiameter;
					
					if baleFillType == "straw" then
						baleWeight = baleVolume * 160;
					elseif baleFillType == "grass_windrow" then
						baleWeight = baleVolume * 200;
					elseif baleFillType == "dryGrass_windrow" then
						baleWeight = baleVolume * 140;
					elseif baleFillType == "silage" then
						baleWeight = baleVolume * 600;
					end;
				else
					local baleWidth = Utils.round(Utils.getNoNil(tonumber(bale.baleWidth), 1.2), 2);
					local baleHeight = Utils.round(Utils.getNoNil(tonumber(bale.baleHeight), 0.9), 2);
					local baleLength = Utils.round(Utils.getNoNil(tonumber(bale.baleLength), 2.4), 2);
					
					--print("current Bale: fillType = " .. baleFillType .. " bale width = " .. baleWidth .. " bale height = " .. baleHeight .. " bale length = " .. baleLength .. " isRoundbale = " .. tostring(isRoundbale));
					
					baleVolume = baleWidth * baleHeight * baleLength;
				
					if baleFillType == "straw" then
						baleWeight = baleVolume * 160;
					elseif baleFillType == "grass_windrow" then                     
						baleWeight = baleVolume * 200;
					elseif baleFillType == "dryGrass_windrow" then
						baleWeight = baleVolume * 140;
					elseif baleFillType == "silage" then                            
						baleWeight = baleVolume * 600;
					end;
				end;
				
				self.fillWeight.allowedAddWeight = false;
				
				newWeight = baleWeight + math.random(20, 80);
			end;
		end;
	else
		self.fillWeight.allowedAddWeight = true;
	end;
	
	return newWeight;
end;

function FillWeight:allowRenderText()
	local allowRenderText = self:getCapacity() > 0 
		and not SpecializationUtil.hasSpecialization(Baler, self.specializations)
		and not self.fillWeight.disableVehicle 
		and self.ultimaTurnOnRotNodes == nil
	;
	
	if g_gui.currentGui ~= nil or g_currentMission.isPlayerFrozen then
       return false;
    end;
    
	if self.isEntered then
		return allowRenderText and self.selectedImplement == nil;
    end;
    
	if self.attacherVehicle ~= nil then
        return allowRenderText and self.isSelected;
    end;
	
	return false;
end;

function FillWeight:getSaveAttributesAndNodes(nodeIdent)
	if self.fillWeight.savedBaleWeight > 0 then
		--## only save the variable 'self.fillWeight.savedBaleWeight' if it is necessary to not mess up the 'vehicles.xml'!
		
		return 'savedBaleWeight="' .. self.fillWeight.savedBaleWeight .. '"', nil;
	end;
	
	return nil, nil;
end;

--## unused but needed functions

function FillWeight:delete()end;
function FillWeight:mouseEvent(posX, posY, isDown, isUp, button)end;
function FillWeight:keyEvent(unicode, sym, modifier, isDown)end;
function FillWeight:update(dt)end;
function FillWeight:updateTick(dt)end;