--[[
	en: 'Lock steering axle' Specialization for Farming Simulator 17
	en: This script allows you to lock an unlock the steering axle
	
	de: 'Lenkachse sperren' Spezialisierung für den Landwirtschafts Simulator 17
	de: Dieses Script erlaubt dir die Lenkachse zu sperren und zu entsperren.
	
	Author: 	Ifko[nator]
	Date:		13.09.2017
	Version:	1.8
	
	History:	v1.0 @ 07.05.2017 - intial implementation in FS 17
				v1.5 @ 10.05.2017 - add save function and support for the old wheel entry
									if the steering axle is unlocked, it will steer also in backwards direction
				v1.6 @ 13.05.2017 - fix for MP
				v1.7 @ 20.08.2017 - fix for joining players
				v1.8 @ 13.09.2017 - again fix for MP
]]

LockSteeringAxles = {};

function LockSteeringAxles.prerequisitesPresent(specializations)
	return SpecializationUtil.hasSpecialization(Attachable, specializations);
end;

function LockSteeringAxles:load(savegame)
	if #self.wheels == 0 then
		return;
	end;
	
	self.foundSteeringAxle = false;
	
	local wheelId = Utils.getNoNil(self.configurations["wheel"], 1);
    local configKey = string.format("vehicle.wheelConfigurations.wheelConfiguration(%d)", wheelId - 1);
	
	if not hasXMLProperty(self.xmlFile, "vehicle.wheelConfigurations.wheelConfiguration(0)") then
		--## fallback to the old wheel entry
		
		configKey = "vehicle";
	end;

    local key = configKey .. ".wheels";
	
	local i = 0;
	
    while true do
        local wheelnamei = key .. string.format(".wheel(%d)", i);	
		
		if not hasXMLProperty(self.xmlFile, wheelnamei) then
			--print("The Vehicle: '" .. self.configFileName .. "' has no steering axle in the 'wheel' TAG. Stop searching now.");
			
			break;
		end;
		
		local steeringAxleRotMax = Utils.getNoNil(getXMLFloat(self.xmlFile, wheelnamei .. "#steeringAxleRotMax"), 0);
	
		if steeringAxleRotMax ~= 0 then
			--print("Found steering axle in line: '" .. i + 1 .. "' from the 'wheel' TAG. Stop searching now.");
			
			self.foundSteeringAxle = true;
			self.steeringAxleUpdateBackwards = true;
			
			break;
		else
			--print("Not found steering axle in line: '" .. i + 1 .. "' from the 'wheel' TAG. Searching now the next line.");
		end;
		
		i = i + 1;
	end;
	
	self.setSteeringAxleActive = LockSteeringAxles.setSteeringAxleActive;
	
	self.lockSteeringAxle = false;
	self.steerAxleIsLocked = false;
	
	if savegame ~= nil and not savegame.resetVehicles then
		self.lockSteeringAxle = Utils.getNoNil(getXMLBool(savegame.xmlFile, savegame.key .. "#lockSteeringAxle"), false);
	end;
	
	self:setSteeringAxleActive(self.lockSteeringAxle, false);
end;

function LockSteeringAxles:writeStream(streamId, connection)
	if self.setSteeringAxleActive ~= nil then	
		streamWriteBool(streamId, self.steerAxleIsLocked);
	end;
end;

function LockSteeringAxles:readStream(streamId, connection)
	if self.setSteeringAxleActive ~= nil then	
		local steerAxleIsLocked = streamReadBool(streamId);

		self:setSteeringAxleActive(steerAxleIsLocked, true);
	end;
end;

function LockSteeringAxles:update(dt)
	if self.isClient and self.foundSteeringAxle then
		if self:getIsActiveForInput() then
			if InputBinding.hasEvent(InputBinding.TOGGLE_CHOPPER) then
				self:setSteeringAxleActive(not self.lockSteeringAxle, false);
			end;
		end;
		
		self.updateSteeringAxleAngle = not self.lockSteeringAxle;
		
		if self.lockSteeringAxle then
			if self.steeringAxleAngle > 0 then
				self.steeringAxleAngle = math.max(self.steeringAxleAngle - self.steeringAxleAngleSpeed * dt, 0);
			else
				self.steeringAxleAngle = math.min(self.steeringAxleAngle + self.steeringAxleAngleSpeed * dt, 0);
			end;
		end;
	end;
end;

function LockSteeringAxles:setSteeringAxleActive(lockSteeringAxle, noEventSend)
	if lockSteeringAxle ~= self.lockSteeringAxle then
		self.lockSteeringAxle = lockSteeringAxle;
		
		LockSteeringAxlesEvent.sendEvent(self, lockSteeringAxle, noEventSend);
	end;
end;

function LockSteeringAxles:draw()
	if self.isClient and self:getIsActiveForInput() and self.foundSteeringAxle then
		if self.lockSteeringAxle then
			g_currentMission:addHelpButtonText(g_i18n:getText("UNLOCK_STEERING_AXLE"), InputBinding.TOGGLE_CHOPPER);
		else
			g_currentMission:addHelpButtonText(g_i18n:getText("LOCK_STEERING_AXLE"), InputBinding.TOGGLE_CHOPPER);
		end;
	end;
end;

function LockSteeringAxles:getSaveAttributesAndNodes(nodeIdent)
	if self.foundSteeringAxle then
		return 'lockSteeringAxle="' .. tostring(self.lockSteeringAxle) .. '"', nil;
	else
		return nil, nil;
	end;
end;

--## unused but needed functions

function LockSteeringAxles:delete()end;
function LockSteeringAxles:mouseEvent(posX, posY, isDown, isUp, button)end;
function LockSteeringAxles:keyEvent(unicode, sym, modifier, isDown)end;
function LockSteeringAxles:updateTick(dt)end;

--## MP Stuff

--## turn steering axle on/off

LockSteeringAxlesEvent = {};
LockSteeringAxlesEvent_mt = Class(LockSteeringAxlesEvent, Event);

InitEventClass(LockSteeringAxlesEvent, "LockSteeringAxlesEvent");

function LockSteeringAxlesEvent:emptyNew()
	local self = Event:new(LockSteeringAxlesEvent_mt);
    
	return self;
end;

function LockSteeringAxlesEvent:new(trailer, lockSteeringAxle)
	local self = LockSteeringAxlesEvent:emptyNew();
	
	self.trailer = trailer;
	self.steerAxleIsLocked = lockSteeringAxle;
	
	return self;
end;

function LockSteeringAxlesEvent:readStream(streamId, connection)
	local id = streamReadInt32(streamId);
	
	self.trailer = networkGetObject(id);
	self.steerAxleIsLocked = streamReadBool(streamId);
    self:run(connection);
end;

function LockSteeringAxlesEvent:writeStream(streamId, connection)
	streamWriteInt32(streamId, networkGetObjectId(self.trailer));
	streamWriteBool(streamId, self.steerAxleIsLocked);
end;

function LockSteeringAxlesEvent:run(connection)
	self.trailer:setSteeringAxleActive(self.steerAxleIsLocked, true);
	
	if not connection:getIsServer() then
		g_server:broadcastEvent(LockSteeringAxlesEvent:new(self.trailer, self.steerAxleIsLocked), nil, connection, self.trailer);
	end;
end;

function LockSteeringAxlesEvent.sendEvent(trailer, lockSteeringAxle, noEventSend)
	if noEventSend == nil or noEventSend == false then
		if g_server ~= nil then
			g_server:broadcastEvent(LockSteeringAxlesEvent:new(trailer, lockSteeringAxle), nil, nil, trailer);
		else
			g_client:getServerConnection():sendEvent(LockSteeringAxlesEvent:new(trailer, lockSteeringAxle));
		end;
	end;
end;