--[[
BaleLoaderExtension
Specialization for extended bale loader functions.

Author:		Ifko[nator] (Kevin Ifkovits)
Datum:		02.08.2017

Version:	v2.6

History:	v1.0 @ 03.09.2016 - initial implementation
			------------------------------------------------------------------------------------------------------------------
			v1.5 @ 29.09.2016 - MP Fix for change grab distance
			------------------------------------------------------------------------------------------------------------------
			v2.0 @ 06.11.2016 - convert to FS 17 and added some new features
			------------------------------------------------------------------------------------------------------------------
			v2.1 @ 22.11.2016 - MP Fix for Error:

								Warning (LUA): 'streamWriteInt32': Argument 1 has wrong type. Expected: Int. Actual: Nil 
								..\..\..\src\base\scripting\lua\LuaScriptSystem.cpp (604): expectedType == Value::VoidType
								LUA call stack:
								=E:/FS 17/mods/FS17_Arcusin_Autostack_Pack/BaleLoaderExtension.lua (470) : streamWriteInt32
								=dataS/scripts/network/Connection.lua (0) : writeStream
								=E:/FS 17/mods/FS17_Arcusin_Autostack_Pack/BaleLoaderExtension.lua (487) : sendEvent
								=E:/FS 17/mods/FS17_Arcusin_Autostack_Pack/BaleLoaderExtension.lua (330) : sendEvent
								=E:/FS 17/mods/FS17_Arcusin_Autostack_Pack/BaleLoaderExtension.lua (59) : setGrabRange
								=dataS/scripts/vehicles/Vehicle.lua (0)
								=dataS/scripts/vehicles/Vehicle.lua (0) : load
								=dataS/scripts/network/Client.lua (0) : readStream
								=dataS/scripts/network/ConnectionManager.lua (0) : func
								
							  - add function to change rim- and axis color via designConfiguration
							  - deactivate Balecollision in an MP Game - to ensure the Bales will be smoothly unloaded 
			------------------------------------------------------------------------------------------------------------------
			v2.2 @ 23.11.2016 - Fix Error:
								
								=E:/FS 17/mods/FS17_Arcusin_Autostack_Pack/BaleLoaderExtension.lua (284) : setScale
								=dataS/scripts/vehicles/Vehicle.lua (0) : update
								=dataS/scripts/network/Server.lua (0) : update
								=dataS/scripts/BaseMission.lua (0) : update
								=dataS/scripts/FSBaseMission.lua (0) : update
								=dataS/scripts/missions/mission00.lua (0) : update
								=dataS/scripts/main.lua (0) : update
								Warning (LUA): 'setScale': Argument 1 has wrong type. Expected: Int. Actual: Nil
								..\..\..\src\base\scripting\lua\LuaScriptSystem.cpp (604): expectedType == Value::VoidType
			------------------------------------------------------------------------------------------------------------------
			v2.5 @ 27.02.2017 - deactivated particle system for wrapped bales
							  - added new bale grab particle systems for: grass, hay and straw
			------------------------------------------------------------------------------------------------------------------
			v2.6 @ 02.08.2017 - deactivated dust particle system when ground is wet
							  - removed function to change rim- and axis color via designConfiguration because this handles now the 'AddConfig.lua'
							  - changed xml TAG's
							  - a little bit code optimization
]]

BaleLoaderExtension = {};

function BaleLoaderExtension.prerequisitesPresent(specializations)
	return SpecializationUtil.hasSpecialization(BaleLoader, specializations);
end;

local debugPriority = 0;

local function printError(errorMessage, isWarning)
	local prefix = "Error ";
	
	if isWarning then
		prefix = "Warning ";
	end;
	
	print(prefix .. "from the BaleLoaderExtension.lua: " .. tostring(errorMessage));
end;

local function printDebug(debugMessage, priority, addStringDebug)
	if debugPriority >= priority then
		local prefix = "";
			
		if addStringDebug then
			prefix = "Debug from the BaleLoaderExtension.lua: "
		end;

		print(prefix .. tostring(debugMessage));
	end;
end;

function BaleLoaderExtension:load(savegame)
	self.doStateChange = Utils.appendedFunction(self.doStateChange, BaleLoaderExtension.doStateChangeBaleLoader);

	self.getAllowsChangeGrabDistance = BaleLoaderExtension.getAllowsChangeGrabDistance;
	self.setGrabMode = BaleLoaderExtension.setGrabMode;
	self.setGrabRange = BaleLoaderExtension.setGrabRange;

	self.grabModeIsProfi = true;
	
	if savegame ~= nil and not savegame.resetVehicles then
		self.grabModeIsProfi = Utils.getNoNil(getXMLBool(savegame.xmlFile, savegame.key .. "#grabModeIsProfi"), self.grabModeIsProfi);
	end;
	
	self:setGrabMode(self.grabModeIsProfi, false);
	
	if self.grabModeIsProfi then
		self:setGrabRange(0.23, false);
	else
		self:setGrabRange(4, false);
	end;
	
	self.baleJointEndTrans = Utils.getVectorNFromString(Utils.getNoNil(getXMLString(self.xmlFile, "vehicle.baleLoaderExtensions.baleJoint#endTrans"), "0 0"), 2);

	self.rolls = {};

	local rollNumber = 0;
	
	printDebug("", 1, false);
	printDebug("-----------------------------------------------Debug from the BaleLoaderExtension.lua Start------------------------------------------------", 1, false);
	printDebug("", 1, false);
	printDebug("-----------------------------------------------Debug from the rolls Start------------------------------------------------", 1, false);
	
	while true do
		local rollKey = "vehicle.baleLoaderExtensions.rolls.roll(" .. tostring(rollNumber) .. ")";

		if not hasXMLProperty(self.xmlFile, rollKey) then
			printDebug("-----------------------------------------------Debug from the rolls End------------------------------------------------", 1, false);
			printDebug("", 1, false);
			
			break;
		end;

		local node = Utils.indexToObject(self.components, getXMLString(self.xmlFile, rollKey .. "#node"));

		if node ~= nil then
			local roll = {};

			roll.node = node;
			roll.isBaleGrabberRoll = Utils.getNoNil(getXMLBool(self.xmlFile, rollKey .. "#isBaleGrabberRoll"), false);
			roll.speed = Utils.getNoNil(getXMLFloat(self.xmlFile, rollKey .. "#speed"), 0.012);
			
			printDebug("Load roll number " .. rollNumber + 1 .. " successfully.", 1, true);

			table.insert(self.rolls, roll);
		else
			printError("Missing the 'node' attribute for the roll number '" .. rollNumber + 1 .. "'! The script will skip this roll now!", false);
		end;

		rollNumber = rollNumber + 1;
	end;

	self.baleGrabberRotNode =  Utils.indexToObject(self.components, getXMLString(self.xmlFile, "vehicle.baleLoaderExtensions.baleGrabber#rotNode"));
	self.baleGrabberEndRot = Utils.getVectorNFromString(Utils.getNoNil(getXMLString(self.xmlFile, "vehicle.baleLoaderExtensions.baleGrabber#endRot"), "0 0 0"), 3);

	self.baleCollision = Utils.indexToObject(self.components, getXMLString(self.xmlFile, "vehicle.baleLoaderExtensions.colllisions#node"));

	if self.baleCollision ~= nil then
		if not g_currentMission.missionDynamicInfo.isMultiplayer then
			self.collisionMask = getCollisionMask(self.baleCollision);

			self.collisionTranslations = {};

			printDebug("-----------------------------------------------Debug from the bale collisions Start------------------------------------------------", 2, false);
			
			for number = 1, #self.balePlaces do
				self.collisionTranslations[number] = Utils.getVectorNFromString(Utils.getNoNil(getXMLString(self.xmlFile, "vehicle.baleLoaderExtensions.colllisions.colllision(" .. tostring(number - 1) .. ")#trans"), "0 0.569 5.359"), 3);
				
				printDebug("Load collision number " .. number .. " value 1 = " .. self.collisionTranslations[number][1] .. " value 2 = " .. self.collisionTranslations[number][2] .. " value 3 = " .. self.collisionTranslations[number][3] .. " successfully.", 2, true);
				
				if number == #self.balePlaces then
					printDebug("-----------------------------------------------Debug from the bale collisions End------------------------------------------------", 2, false);
					printDebug("", 2, false);
				end;
			end;
		else
			setScale(self.baleCollision, 0, 0, 0);
		end;
	else
		printError("Missing the 'node' attribute for the bale collision! The script will skip the bale collision now!", true);
	end;

	if self.isClient then
		self.workSounds = {};

		local soundNumber = 0;

		printDebug("-----------------------------------------------Debug from the work sounds Start------------------------------------------------", 1, false);
		
		while true do
			local soundKey = "vehicle.baleLoaderExtensions.workSounds.workSound(" .. tostring(soundNumber) .. ")";

			if not hasXMLProperty(self.xmlFile, soundKey) then
				printDebug("-----------------------------------------------Debug from the work sounds End------------------------------------------------", 1, false);
				printDebug("", 1, false);
				
				break;
			end;

			local workSound = SoundUtil.loadSample(self.xmlFile, {}, soundKey, nil, self.baseDirectory, self.baleGrabberRotNode);

			if workSound ~= nil then
				printDebug("Load work sound number " .. soundNumber + 1 .. " successfully.", 1, true);
			
				table.insert(self.workSounds, workSound);
			else
				printError("Missing the 'workSound' attribute for the sound number '" .. soundNumber + 1 .. "'! The script will skip this sound now!", false);
			end;

			soundNumber = soundNumber + 1;
		end;
	end;

	self.baleGrabParticleSystems = {};

	local particleSystemNumber = 0;

	printDebug("-----------------------------------------------Debug from the bale grab particle systems Start------------------------------------------------", 1, false);
	
	while true do
		local particleSystemKey = "vehicle.baleLoaderExtensions.baleGrabParticleSystems.baleGrabParticleSystem(" .. tostring(particleSystemNumber) .. ")";
		
		if not hasXMLProperty(self.xmlFile, particleSystemKey) then
			printDebug("-----------------------------------------------Debug from the bale grab particle systems End------------------------------------------------", 1, false);
			printDebug("", 1, false);
			
			break;
		end;
		
		local node = Utils.indexToObject(self.components, getXMLString(self.xmlFile, particleSystemKey .. "#node"));
		
		if node ~= nil then
			local baleGrabParticleSystem = {};
			
			baleGrabParticleSystem.particleSystem = {};
			
			baleGrabParticleSystem.disableDuration = Utils.getNoNil(getXMLFloat(self.xmlFile, particleSystemKey .. "#disableDuration"), 0.6) * 1000;
			baleGrabParticleSystem.activeAtBaleGrabbing = Utils.getNoNil(getXMLBool(self.xmlFile, particleSystemKey .. "#activeAtBaleGrabbing"), false);
			baleGrabParticleSystem.onlyActiveAtBaleGrabbing = Utils.getNoNil(getXMLBool(self.xmlFile, particleSystemKey .. "#onlyActiveAtBaleGrabbing"), false);
			baleGrabParticleSystem.activeForBaleFillType = Utils.getNoNil(getXMLString(self.xmlFile, particleSystemKey .. "#activeForBaleFillType"), "straw");
			baleGrabParticleSystem.activeDuringRain = Utils.getNoNil(getXMLBool(self.xmlFile, particleSystemKey .. "#activeDuringRain"), true);
			baleGrabParticleSystem.disableTime = 0;
			
			printDebug("Load work bale grab particle system number " .. particleSystemNumber + 1 .. " for fillType " .. baleGrabParticleSystem.activeForBaleFillType .. " successfully.", 1, true);
			
			ParticleUtil.loadParticleSystem(self.xmlFile, baleGrabParticleSystem.particleSystem, particleSystemKey, self.components, false, nil, self.baseDirectory);
			
			table.insert(self.baleGrabParticleSystems, baleGrabParticleSystem);
		else
			printError("Missing the 'node' attribute for the bale grab particle system number '" .. particleSystemNumber + 1 .. "'! The script will skip this bale grab particle system now!", false);
		end;
		
		particleSystemNumber = particleSystemNumber + 1;
	end;
	
	printDebug("-----------------------------------------------Debug from the BaleLoaderExtension.lua End------------------------------------------------", 1, false);
	printDebug("", 1, false);
	
	self.baleloaderTurnedOnScrollersModdingWelt = Utils.loadScrollers(self.components, self.xmlFile, "vehicle.baleLoaderExtensions.baleloaderTurnedOnScrollers.baleloaderTurnedOnScroller", {}, false);
end;

function BaleLoaderExtension:doStateChangeBaleLoader(id, nearestBaleServerId)
	if id == BaleLoader.CHANGE_GRAB_BALE then
		--## play animation for pick up the bale to the bale grabber.
		self:playAnimation("baleGrabberLoadBale", 1, nil, true);
	elseif id == BaleLoader.CHANGE_GRAB_MOVE_DOWN then
		--## reset bale animation to start values for the next bale!
		self:playAnimation("baleGrabberLoadBale", -5000, nil, true);
	end;
end;

function BaleLoaderExtension:getAllowsChangeGrabDistance()
	return self.emptyState == BaleLoader.EMPTY_NONE and self.isInWorkPosition;
end;

function BaleLoaderExtension:update(dt)
	if self:getIsActive() or not g_firstRunIsDone then
		g_firstRunIsDone = true;
		
		local grabberRotX, grabberRotY, grabberRotZ = getRotation(self.baleGrabberRotNode);
		local _, baleTransY, baleTransZ = getTranslation(self.baleGrabber.grabNode);
		
		grabberRotX, grabberRotY, grabberRotZ = Utils.round(math.deg(grabberRotX), 2), Utils.round(math.deg(grabberRotY), 2), Utils.round(math.deg(grabberRotZ), 2);
		baleTransY, baleTransZ = Utils.round(baleTransY, 2), Utils.round(baleTransZ, 2);
		
		local isPickUpBale = baleTransY ~= self.baleJointEndTrans[1] and baleTransZ ~= self.baleJointEndTrans[2] and self.grabberMoveState == BaleLoader.GRAB_MOVE_UP;
		local isLoadingBaleToPlatform = grabberRotX == self.baleGrabberEndRot[1] and grabberRotY == self.baleGrabberEndRot[2] and grabberRotZ == self.baleGrabberEndRot[3] and self.grabberMoveState == BaleLoader.GRAB_DROP_BALE;
		
		local baleWrappingState = 0;
		local baleFillType = "unknown";
		
		if self.baleGrabber.currentBale ~= nil then
			local bale = networkGetObject(self.baleGrabber.currentBale);
			
			if bale ~= nil then
				baleWrappingState = Utils.getNoNil(bale.wrappingState, baleWrappingState);
				baleFillType = Utils.getNoNil(FillUtil.fillTypeIntToName[bale.fillType], baleFillType);
			end;
		end;
		
		--setTextAlignment(RenderText.ALIGN_CENTER);
		--renderText(0.5, 0.5, 0.02, "current bale fill type at the bale grabber = " .. baleFillType);
		
		if isPickUpBale then
			--## pick bale from ground up
			
			if self.rolls ~= nil then
				for _, roll in pairs(self.rolls) do
					local rollRotX, _, _ = getRotation(roll.node);
					local speed = roll.speed * dt;
					
					if roll.isBaleGrabberRoll then
						rollRotX = rollRotX - speed;

						setRotation(roll.node, rollRotX, 0, 0);
					end;
				end;
			end;
			
			if self.baleGrabParticleSystems ~= nil then
				for _, baleGrabParticleSystem in pairs(self.baleGrabParticleSystems) do
					if (baleGrabParticleSystem.activeAtBaleGrabbing or baleGrabParticleSystem.onlyActiveAtBaleGrabbing) and baleWrappingState == 0 and baleFillType == baleGrabParticleSystem.activeForBaleFillType	then
						if baleGrabParticleSystem.activeDuringRain then	
							ParticleUtil.setEmittingState(baleGrabParticleSystem.particleSystem, true);
						else
							ParticleUtil.setEmittingState(baleGrabParticleSystem.particleSystem, g_currentMission.environment.groundWetness < 0.1);
						end;
						
						baleGrabParticleSystem.disableTime = g_currentMission.time + baleGrabParticleSystem.disableDuration;
					end;
				end;
			end;
		elseif isLoadingBaleToPlatform then
			--## load bale from bale grabber to bale platform
		
			if self.rolls ~= nil then
				for _, roll in pairs(self.rolls) do
					local rollRotX, _, rollRotZ = getRotation(roll.node);
					local speed = roll.speed * dt;
					
					if roll.isBaleGrabberRoll then
						rollRotX = rollRotX - speed;
					else
						rollRotZ = rollRotZ - speed;
					end;
					
					setRotation(roll.node, rollRotX, 0, rollRotZ);
				end;
			end;
		
			if self.baleGrabParticleSystems ~= nil then
				for _, baleGrabParticleSystem in pairs(self.baleGrabParticleSystems) do
					if not baleGrabParticleSystem.onlyActiveAtBaleGrabbing and baleWrappingState == 0 and baleFillType == baleGrabParticleSystem.activeForBaleFillType then	
						if baleGrabParticleSystem.activeDuringRain then	
							ParticleUtil.setEmittingState(baleGrabParticleSystem.particleSystem, true);
						else
							ParticleUtil.setEmittingState(baleGrabParticleSystem.particleSystem, g_currentMission.environment.groundWetness < 0.1);
						end;
					
						baleGrabParticleSystem.disableTime = g_currentMission.time + baleGrabParticleSystem.disableDuration;
					end;
				end;
			end;
		end;
		
		if isPickUpBale or isLoadingBaleToPlatform then
			if self.workSounds ~= nil then
				for _, workSound in pairs(self.workSounds) do	
					if self:getIsActiveForSound() then	
						SoundUtil.playSample(workSound, 0, 0, nil);
					else --if self.isControlled then
						SoundUtil.play3DSample(workSound);
					end;
				end;
			end
		else
			if self.workSounds ~= nil then
				for _, workSound in pairs(self.workSounds) do
					SoundUtil.stopSample(workSound, true);
					SoundUtil.stop3DSample(workSound);
				end;
			end;
			
			if self.isClient and self.baleGrabParticleSystems ~= nil then 
				for _, baleGrabParticleSystem in pairs(self.baleGrabParticleSystems) do
					if baleGrabParticleSystem.disableTime ~= 0 and baleGrabParticleSystem.disableTime < g_currentMission.time then
						ParticleUtil.setEmittingState(baleGrabParticleSystem.particleSystem, false);
					
						baleGrabParticleSystem.disableTime = 0;
					end;
				end;
			end;
		end;
		
		Utils.updateScrollers(self.baleloaderTurnedOnScrollersModdingWelt, dt, isPickUpBale or isLoadingBaleToPlatform);
		
		if not g_currentMission.missionDynamicInfo.isMultiplayer and self.baleCollision ~= nil then
			local scaleX, scaleY, scaleZ = 1, 1, 1;
			local baleRows = 1;
			
			if self:getFillLevel() == 0 then
				scaleX, scaleY, scaleZ = 0, 0, 0;
			elseif self:getFillLevel() >= 2 then
				scaleY = scaleY * 2;
			end;
			
			if self:getFillLevel() >= 3 and self:getFillLevel() < 5 then
				baleRows = 2;
			elseif self:getFillLevel() >= 5 and self:getFillLevel() < 7 then
				baleRows = 3;
			elseif self:getFillLevel() >= 7 and self:getFillLevel() < 9 then
				baleRows = 4;
			elseif self:getFillLevel() >= 9 and self:getFillLevel() < 11 then
				baleRows = 5;
			elseif self:getFillLevel() >= 11 and self:getFillLevel() < 13 then
				baleRows = 6;
			elseif self:getFillLevel() >= 13 and self:getFillLevel() < 15 then
				baleRows = 7;
			elseif self:getFillLevel() >= 15 then
				baleRows = 8;
			end;
			
			setScale(self.baleCollision, scaleX, scaleY, scaleZ * baleRows);
			
			if self:getFillLevel() == 0 then	
				baleRows = #self.balePlaces;
			end;
			
			if self.emptyState == BaleLoader.EMPTY_ROTATE_PLATFORM or self.emptyState == BaleLoader.EMPTY_SINK then
				setTranslation(self.baleCollision, self.collisionTranslations[baleRows][1], self.collisionTranslations[baleRows][2], self.collisionTranslations[baleRows][3]);
			end;
			
			if self.emptyState == BaleLoader.EMPTY_WAIT_TO_SINK then
				setCollisionMask(self.baleCollision, 0);
			elseif self.emptyState == BaleLoader.EMPTY_NONE then
				setCollisionMask(self.baleCollision, self.collisionMask);
			end;
		
			--[[
			renderText(0.5, 0.52, 0.02, "translationX = '" .. self.collisionTranslations[baleRows][1] .. "' translationY = '" .. self.collisionTranslations[baleRows][2] .. "' translationZ = '" .. self.collisionTranslations[baleRows][3] .. "'.");
			renderText(0.5, 0.5, 0.02, "scaleX = '" .. scaleX .. "' scaleY = '" .. scaleY .. "' scaleZ = '" .. scaleZ * baleRows .. "'.");
			renderText(0.5, 0.48, 0.02, "collision mask from the bale collision = " .. getCollisionMask(self.baleCollision));
			renderText(0.5, 0.46, 0.02, "number of collison translations = " .. #self.collisionTranslations);
			]]
		end;
	
		if self:getFillLevel() == self:getCapacity()
			and self.emptyState == BaleLoader.EMPTY_NONE 
			and self.grabberMoveState == nil 
			and self.rotatePlatformDirection == 0 
			and self.isInWorkPosition 
		then
		--[[baleloader is full, 
				baleloader state is 'none', 
				grabber is not moving,
				stackplatform is set down,
				and grabber is in work position 
			- 
				move grabber to transport position
		]]
			g_client:getServerConnection():sendEvent(BaleLoaderStateEvent:new(self, BaleLoader.CHANGE_BUTTON_WORK_TRANSPORT));
		end;
	end;

	if self:getIsActiveForInput() and self:getAllowsChangeGrabDistance() then
		if InputBinding.hasEvent(InputBinding.IMPLEMENT_EXTRA2) then
			self:setGrabMode(not self.grabModeIsProfi, false);
			
			if self.grabModeIsProfi then
				self:setGrabRange(0.23, false);
			else
				self:setGrabRange(4, false);
			end;
		end;
	end;
end;

function BaleLoaderExtension:setGrabMode(grabMode, noEventSend)
	if grabMode ~= self.grabModeIsProfi then
		self.grabModeIsProfi = grabMode;
		
		BaleLoaderExtensionGrabModeEvent.sendEvent(self, grabMode, noEventSend);
	end;
end;

function BaleLoaderExtension:setGrabRange(range, noEventSend)
	if range ~= self.balePickupRange then
		self.balePickupRange = range;
		
		BaleLoaderExtensionGrabRangeEvent.sendEvent(self, range, noEventSend);
	end;
end;

function BaleLoaderExtension:draw()
	if self:getIsActiveForInput() and self:getAllowsChangeGrabDistance() then
		g_currentMission:addHelpButtonText(g_i18n:getText("CANGE_GRAB_MODE"), InputBinding.IMPLEMENT_EXTRA2);
	
		if self.grabModeIsProfi then
			g_currentMission:addExtraPrintText(g_i18n:getText("PROFI_GRAB_MODE"));
		else	
			g_currentMission:addExtraPrintText(g_i18n:getText("ROOKIE_GRAB_MODE"));
		end;
	end;
end;

function BaleLoaderExtension:getSaveAttributesAndNodes(nodeIdent)
	local attributes = 'grabModeIsProfi="' .. tostring(self.grabModeIsProfi) .. '"';
	
	return attributes;
end;

function BaleLoaderExtension:delete()
	if self.isClient and self.workSounds ~= nil then
		for _, workSound in pairs(self.workSounds) do
			SoundUtil.deleteSample(workSound);
		end;
	end;
	
	if self.baleGrabParticleSystems ~= nil then
        for _, baleGrabParticleSystem in pairs(self.baleGrabParticleSystems) do
			ParticleUtil.deleteParticleSystem(baleGrabParticleSystem.particleSystem);
		end;
	end;
end;

function BaleLoaderExtension:onDeactivate()
	if self.isClient and self.workSounds ~= nil then
		for _, workSound in pairs(self.workSounds) do	
			SoundUtil.stopSample(workSound, true);
			SoundUtil.stop3DSample(workSound);
		end;
    end;
	
	if self.baleGrabParticleSystems ~= nil then
        for _, baleGrabParticleSystem in pairs(self.baleGrabParticleSystems) do
			ParticleUtil.setEmittingState(baleGrabParticleSystem.particleSystem, false);
		end;
	end;
end;

--## unused but needed functions

function BaleLoaderExtension:mouseEvent(posX, posY, isDown, isUp, button)end;
function BaleLoaderExtension:keyEvent(unicode, sym, modifier, isDown)end;

--## MP Stuff

--## set grab mode

BaleLoaderExtensionGrabModeEvent = {};
BaleLoaderExtensionGrabModeEvent_mt = Class(BaleLoaderExtensionGrabModeEvent, Event);

InitEventClass(BaleLoaderExtensionGrabModeEvent, "BaleLoaderExtensionGrabModeEvent");

function BaleLoaderExtensionGrabModeEvent:emptyNew()
	local self = Event:new(BaleLoaderExtensionGrabModeEvent_mt);
    
	return self;
end;

function BaleLoaderExtensionGrabModeEvent:new(baleLoader, grabModeIsProfi)
	local self = BaleLoaderExtensionGrabModeEvent:emptyNew();
	
	self.baleLoader = baleLoader;
	self.grabMode = grabModeIsProfi;
	
	return self;
end;

function BaleLoaderExtensionGrabModeEvent:readStream(streamId, connection)
	local id = streamReadInt32(streamId);
	
	self.baleLoader = networkGetObject(id);
	self.grabMode = streamReadBool(streamId);
    self:run(connection);
end;

function BaleLoaderExtensionGrabModeEvent:writeStream(streamId, connection)
	streamWriteInt32(streamId, networkGetObjectId(self.baleLoader));
	streamWriteBool(streamId, self.grabMode);
end;

function BaleLoaderExtensionGrabModeEvent:run(connection)
	self.baleLoader:setGrabMode(self.grabMode, true);
	
	if not connection:getIsServer() then
		g_server:broadcastEvent(BaleLoaderExtensionGrabModeEvent:new(self.baleLoader, self.grabMode), nil, connection, self.baleLoader);
	end;
end;

function BaleLoaderExtensionGrabModeEvent.sendEvent(baleLoader, grabMode, noEventSend)
	if noEventSend == nil or noEventSend == false then
		if g_server ~= nil then
			g_server:broadcastEvent(BaleLoaderExtensionGrabModeEvent:new(baleLoader, grabMode), nil, nil, baleLoader);
		else
			g_client:getServerConnection():sendEvent(BaleLoaderExtensionGrabModeEvent:new(baleLoader, grabMode));
		end;
	end;
end;

-----------------------------------------------------------------------------------------------------------------------------------

--## set grab distance

BaleLoaderExtensionGrabRangeEvent = {};
BaleLoaderExtensionGrabRangeEvent_mt = Class(BaleLoaderExtensionGrabRangeEvent, Event);

InitEventClass(BaleLoaderExtensionGrabRangeEvent, "BaleLoaderExtensionGrabRangeEvent");

function BaleLoaderExtensionGrabRangeEvent:emptyNew()
	local self = Event:new(BaleLoaderExtensionGrabRangeEvent_mt);
   
	return self;
end;

function BaleLoaderExtensionGrabRangeEvent:new(baleLoader, range)
	local self = BaleLoaderExtensionGrabRangeEvent:emptyNew();
	
	self.baleLoader = baleLoader;
	self.range = range;
	
	return self;
end;

function BaleLoaderExtensionGrabRangeEvent:readStream(streamId, connection)
	local id = streamReadInt32(streamId);
	
	self.baleLoader = networkGetObject(id);
	self.range = streamReadFloat32(streamId);
    self:run(connection);
end;

function BaleLoaderExtensionGrabRangeEvent:writeStream(streamId, connection)
	streamWriteInt32(streamId, networkGetObjectId(self.baleLoader));
	streamWriteFloat32(streamId, self.range);
end;

function BaleLoaderExtensionGrabRangeEvent:run(connection)
	self.baleLoader:setGrabRange(self.range, true);
	
	if not connection:getIsServer() then
		g_server:broadcastEvent(BaleLoaderExtensionGrabRangeEvent:new(self.baleLoader, self.range), nil, connection, self.baleLoader);
	end;
end;

function BaleLoaderExtensionGrabRangeEvent.sendEvent(baleLoader, range, noEventSend)
	if noEventSend == nil or noEventSend == false then
		if g_server ~= nil then
			g_server:broadcastEvent(BaleLoaderExtensionGrabRangeEvent:new(baleLoader, range), nil, nil, baleLoader);
		else
			g_client:getServerConnection():sendEvent(BaleLoaderExtensionGrabRangeEvent:new(baleLoader, range));
		end;
	end;
end;