--[[
Specialization for the automatic product loading of the Fliegl DPW.

Source: UBT Script

Author: Ifko[nator]
Date: 19.09.2015
Version: 3.0

History: V 1.0 @ 20.06.2015 - intial release
		 ---------------------------------------------------------------------------------------------------------
		 V 1.5 @ 30.06.2015 - added Couresplay support - Thanks to Thomas Gärtner for helping
		 ---------------------------------------------------------------------------------------------------------
		 V 2.0 @ 09.07.2015 - added support for the ros Transportboxes
		 ---------------------------------------------------------------------------------------------------------
		 v 3.0 @ 19.09.2015 - added the option to choose the product type from the front or back attached trailer
							  added the option to switch forward and backwards between the product types
							  added support for the roundbales and the pallets from the Krone Ultima by BM Modding
							  added support for the pallets from the Mengele Granat by Stefan Maurus
							  added support for the egg pallets by Farmer_Andy
							  added support for the pallets from the greenhouses
							  added support for the H-Milk cans by Farmer_Andy
							  added support for the pallets from Kastor's Obstfarm
							  added support for the big saplings pallets by t0xic0m 
		 ---------------------------------------------------------------------------------------------------------

Website: http://www.modding-welt.com/index.php/BoardList
]]

FlieglMain = {};

function FlieglMain.prerequisitesPresent(specializations)
    if SpecializationUtil.hasSpecialization(Fillable, specializations) then
		return true;
	else
		print("[ERROR from the FlieglMain.lua]: Missing the specialization 'Fillable' in: '" .. self.configFileName .. "'. Script is stopped now!");
		return false;
	end;
end;

function FlieglMain:load(xmlFile)
	self.automaticLoading = true;
	
	self.automaticLoadingText = {
		g_i18n:getText("AUTOLOAD_MANUAL"), 
		g_i18n:getText("AUTOLOAD_AUTOMATIC")
	};
	
	self.loadingIsActive = false;
	self.showFlieglHelp = false;
	
	--Courseplay identifier
	self.isSpecialBaleLoader = true;
	self.isFlieglDPWxxx = true;
	self.unloadingIsActive = false;
	self.isLookingForBales = false;
	
	self.maxSpeedLimit = Utils.getNoNil(getXMLInt(xmlFile, "vehicle.fliegl#maxSpeedLimit"), 25);
	
	self.warningText = {
		g_i18n:getText("TO_UNLOAD_STOP_AUTOLOAD"), 
		g_i18n:getText("TO_CHANGE_UNLOAD_SIDE_STOP_AUTOLOAD"), 
		g_i18n:getText("TO_CHANGE_BALETYPE_UNLOAD"), 
		g_i18n:getText("TO_CHANGE_LOADINGTYPE_STOP_AUTOLOAD"),
		string.format(g_i18n:getText("TO_FAST_DRIVEN_AUTOLOAD_WAS_STOPPED"), self.maxSpeedLimit),
		string.format(g_i18n:getText("YOU_DRIVE_TO_FAST_TO_LOAD"), self.maxSpeedLimit)
	};
	
	self.maxSpeedLimit = self.maxSpeedLimit + 1;
	
	self.warning = {false, false, false, false, false, false, false};
	
	self.unloadLeft = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.fliegl.unload#leftIndex"));
	self.unloadRight = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.fliegl.unload#rightIndex"));
	self.unloadBack = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.fliegl.unload#backIndex"));
	
	self.showUnloadingHelp = false;
	
	self.helpLineLeftStart = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.fliegl.helpLines.left#startIndex"));
	self.helpLineLeftEnd = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.fliegl.helpLines.left#endIndex"));
	self.helpLineLeftStart2 = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.fliegl.helpLines.left#startIndex2"));
	self.helpLineLeftEnd2 = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.fliegl.helpLines.left#endIndex2"));
	
	self.helpLineRightStart = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.fliegl.helpLines.right#startIndex"));
	self.helpLineRightEnd = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.fliegl.helpLines.right#endIndex"));
	self.helpLineRightStart2 = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.fliegl.helpLines.right#startIndex2"));
	self.helpLineRightEnd2 = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.fliegl.helpLines.right#endIndex2"));
	
	self.helpLineBackStart = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.fliegl.helpLines.back#startIndex"));
	self.helpLineBackEnd = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.fliegl.helpLines.back#endIndex"));
	self.helpLineBackStart2 = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.fliegl.helpLines.back#startIndex2"));
	self.helpLineBackEnd2 = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.fliegl.helpLines.back#endIndex2"));
	
	self.colorRed = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.fliegl.helpLines.colors#red"), 0.95);
	self.colorGreen = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.fliegl.helpLines.colors#green"), 0);
	self.colorBlue = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.fliegl.helpLines.colors#blue"), 0);
	
	self.nearestDistance = Utils.getNoNil(getXMLInt(xmlFile, "vehicle.fliegl#nearestDistance"), 10);
	
	self.unloadSide = 1;
	
	self.currentUnloadSide = {
		{g_i18n:getText("UNLOAD_LEFT"), false, self.unloadLeft},
		{g_i18n:getText("UNLOAD_BACK"), false, self.unloadBack},
		{g_i18n:getText("UNLOAD_RIGHT"), false, self.unloadRight},
		{g_i18n:getText("UNLOAD_TRAILER"), false, 0}
	};	
	
	self.productType = 1;
	
	self.numProducts = {};
	self.attacher = {};
	self.attacherLevel = {};
	self.choosenProductType = {};
	
	local numFlieglMainTypes = Utils.getNoNil(getXMLInt(xmlFile, "vehicle.fliegl#numProductTypes"), 0);
	
	for i = 1, numFlieglMainTypes do
		self.numProducts[i] = Utils.getNoNil(getXMLInt(xmlFile, string.format("vehicle.fliegl.product" .. "%d" .. "#numProducts", i)), 0);
		self.choosenProductType[i] = {g_i18n:getText(Utils.getNoNil(getXMLString(xmlFile, string.format("vehicle.fliegl.product" .. "%d" .. "#type", i))), "nil"), i};
		self.attacher[i] = {};
		self.attacherLevel[i] = Utils.indexToObject(self.components, getXMLString(xmlFile, string.format("vehicle.fliegl.product" .. "%d" .. "#index", i)));
		for j = 1, self.numProducts[i] do
			self.attacher[i][j] = {};
			local partName = string.format("vehicle.fliegl.product" .. "%d" .. ".productAttacher" .. "%d", i, j);
			self.attacher[i][j].object = Utils.indexToObject(self.components, getXMLString(xmlFile, partName .. "#index"));
			self.attacher[i][j].attachedProduct = nil;
			self.attacher[i][j].productInRange = nil;
		end;
	end;
	
	self.testMode = Utils.getNoNil(getXMLBool(xmlFile, "vehicle.fliegl#testMode"), false);
	
    self.capacity = self.numProducts[self.productType];
end;

function FlieglMain:loadFromAttributesAndNodes(xmlFile, key, resetVehicles)
	if not resetVehicles then
		local productType = Utils.getNoNil(getXMLInt(xmlFile, key .. "#productType"), self.productType);
		local unloadSide = Utils.getNoNil(getXMLInt(xmlFile, key .. "#unloadSide"), self.unloadSide);
		local automaticLoading = Utils.getNoNil(getXMLBool(xmlFile, key.. "#automaticLoading"), self.automaticLoading);
		local showFlieglHelp = Utils.getNoNil(getXMLBool(xmlFile, key .. "#showFlieglHelp"), self.showFlieglHelp);
		local showUnloadingHelp = Utils.getNoNil(getXMLBool(xmlFile, key .. "#showUnloadingHelp"), self.showUnloadingHelp);
	
		self.productType = productType;
		self.capacity = self.numProducts[self.productType];
		self.fillLevel = 0;
		self.unloadSide = unloadSide;
		self.automaticLoading = automaticLoading;
		self.showFlieglHelp = showFlieglHelp;
		self.showUnloadingHelp = showUnloadingHelp;
	end; 

    return BaseMission.VEHICLE_LOAD_OK;
end;

function FlieglMain:getSaveAttributesAndNodes(nodeIdent)
	self.backMode = self.currentUnloadSide[self.unloadSide][1];
	
	if self.fillLevel > 0 then
		self.currentUnloadSide[self.unloadSide][1] = g_i18n:getText("UNLOAD_TRAILER");
		
		for i = 1, self.fillLevel do
			if self.attacher[self.productType][i].attachedProduct ~= nil then
				FlieglMain:ProductDetachFlieglMain(self, i);
				self.fillLevel = self.fillLevel - 1;
			end;
		end;
		self.fillLevel = 0;
	end;
	self.currentUnloadSide[self.unloadSide][1] = self.backMode
	
	if self.loadingIsActive then
		self.loadingIsActive = false;
	end;
	
	local attributes = 
		'productType="' .. tostring(self.productType) ..
		'" unloadSide="' .. tostring(self.unloadSide) ..
		'" automaticLoading="' .. tostring(self.automaticLoading) ..
		'" showFlieglHelp="' .. tostring(self.showFlieglHelp) ..
		'" showUnloadingHelp="' .. tostring(self.showUnloadingHelp) ..
	'"';
	
	return attributes, nil;
end;


function FlieglMain:delete()
	self.currentUnloadSide[self.unloadSide][1] = g_i18n:getText("UNLOAD_TRAILER");
	
	if self.fillLevel > 0 then
		for i = 1, self.fillLevel do
			if self.attacher[self.productType][i].attachedProduct ~= nil then
				FlieglMain:ProductDetachFlieglMain(self, i);
				self.fillLevel = self.fillLevel - 1;
			end;
		end;
		self.fillLevel = 0;
	end;
	
	if self.loadingIsActive then
		self.loadingIsActive = false;
	end;
end;

function FlieglMain:mouseEvent(posX, posY, isDown, isUp, button)
end;

function FlieglMain:keyEvent(unicode, sym, modifier, isDown)
end;

function FlieglMain:update(dt)
	self.warning = {false, false, false, false, false, false, false};
	
	local maxSpeedLimit = self.maxSpeedLimit/3600;
	
	if self.lastSpeedReal > maxSpeedLimit and self.loadingIsActive then
		self.loadingIsActive = false;
		self.warning[7] = true;
		self.warning[5] = true;
	end;
	
	if self:getIsActiveForInput() or self.unloadingIsActive then
		if InputBinding.hasEvent(InputBinding.FLIEGL_LOAD) then 
			if not self.automaticLoading then
				if self.lastSpeedReal < maxSpeedLimit then
					for i = 1, self.numProducts[self.productType] do
						if self.attacher[self.productType][i].attachedProduct == nil then
							local nearest = FlieglMain:ProductInRangeFlieglMain(self, i);
							if nearest ~= nil then
								self.attacher[self.productType][i].productInRange = nearest;
								FlieglMain:ProductAttachFlieglMain(self, i);
							else
								break;
							end;
						end;
					end;
				else
					self.warning[7] = true;
					self.warning[6] = true;
				end;
			else
				self.loadingIsActive = not self.loadingIsActive;
			end;
		end;
		
		if InputBinding.hasEvent(InputBinding.FLIEGL_UNLOAD) or self.unloadingIsActive then
			if not self.loadingIsActive then
				for k = 1, self.numProducts[self.productType] do
					if self.attacher[self.productType][k].attachedProduct ~= nil then
						FlieglMain:ProductDetachFlieglMain(self, k);
						self.fillLevel = self.fillLevel - 1;
					end;
				end;
				self.fillLevel = 0;
			else
				self.warning[7] = true;
				self.warning[1] = true;
			end;
			
			self.unloadingIsActive = false;
		end;
		
		if InputBinding.hasEvent(InputBinding.FLIEGL_TOGGLE_UNLOAD_SIDE) then
			if not self.loadingIsActive then
				if self.unloadSide == 4 then
					self.unloadSide = 0;
				end;
				
				self.unloadSide = self.unloadSide + 1;
			else
				self.warning[7] = true;
				self.warning[2] = true;
			end;
		end;
		
		if InputBinding.hasEvent(InputBinding.FLIEGL_TOGGLE_PRODUCT_TYPE_UP) then
			if self.fillLevel == 0 then
				if self.productType == #(self.numProducts) then
					self.productType = 0;
				end;
				
				self.productType = self.productType + 1;
				self.capacity = self.numProducts[self.productType];
			else
				self.warning[7] = true;
				self.warning[3] = true;
			end;
		elseif InputBinding.hasEvent(InputBinding.FLIEGL_TOGGLE_PRODUCT_TYPE_DOWN) then
			if self.fillLevel == 0 then
				if self.productType == 1 then
					self.productType = #(self.numProducts) + 1;
				end; 
				
				self.productType = self.productType - 1;
				self.capacity = self.numProducts[self.productType];
			else
				self.warning[7] = true;
				self.warning[3] = true;
			end;
		end;
		
		if self.fillLevel == 0 then
			if self.attacherVehicle ~= nil and self.attacherVehicle:getIsActive() and self.attacherVehicle.productType ~= nil and self.attacherVehicle.productType ~= self.productType then
				if InputBinding.hasEvent(InputBinding.FLIEGL_TOGGLE_PRODUCT_FROM_FRONT_ATTACHED_TRAILER) then
					self.productType = self.attacherVehicle.productType;
					self.capacity = self.numProducts[self.productType];
				end;
			end;
			
			for _, attachedTrailer in pairs(self.attachedImplements) do
				if attachedTrailer.object ~= nil then
					local backAttachedTrailer = attachedTrailer.object;
					
					if backAttachedTrailer:getIsActive() and backAttachedTrailer.productType ~= nil and backAttachedTrailer.productType ~= self.productType then
						if InputBinding.hasEvent(InputBinding.FLIEGL_TOGGLE_PRODUCT_FROM_BACK_ATTACHED_TRAILER) then
							self.productType = backAttachedTrailer.productType;
							self.capacity = self.numProducts[self.productType];
						end;
					end;
				end;
			end;
		end;
			
		if InputBinding.hasEvent(InputBinding.FLIEGL_TOGGLE_LOADING_TYPE) then
			if not self.loadingIsActive then
				self.automaticLoading = not self.automaticLoading
			else
				self.warning[7] = true;
				self.warning[4] = true;
			end;
		end;

		if InputBinding.hasEvent(InputBinding.FLIEGL_TOGGLE_HELP) then
			self.showFlieglHelp = not self.showFlieglHelp;
		end;
		
		if InputBinding.hasEvent(InputBinding.FLIEGL_TOGGLE_UNLOAD_HELP) then
			self.showUnloadingHelp = not self.showUnloadingHelp;
		end;
	end;
end;

function FlieglMain:updateTick(dt)
	if self.fillLevel == self.capacity and self.loadingIsActive then
		self.loadingIsActive = false;
	end;
	
	local i = self.fillLevel + 1;
	
	if self.automaticLoading and (self.loadingIsActive or self.isLookingForBales) then
		self.nearestObject = FlieglMain:ProductInRangeFlieglMain(self, i);
	end
	
	if self.automaticLoading and self.loadingIsActive then		
		if self.attacher[self.productType][i].attachedProduct == nil then
			if self.nearestObject ~= nil then
				self.attacher[self.productType][i].productInRange = self.nearestObject;
				FlieglMain:ProductAttachFlieglMain(self, i);
			end;
		end;
	end;
end;

function FlieglMain:draw()
	local autoLoadText = "unknown";
	local warningText = "unknown";
	
	if not self.automaticLoading then
		autoLoadText = self.automaticLoadingText[1];
	else
		autoLoadText = self.automaticLoadingText[2];
	end;
	
	if self.warning[1] then
		warningText = self.warningText[1];
	elseif self.warning[2] then
		warningText = self.warningText[2];
	elseif self.warning[3] then
		warningText = self.warningText[3];
	elseif self.warning[4] then
		warningText = self.warningText[4];
	elseif self.warning[5] then
		warningText = self.warningText[5];
	elseif self.warning[6] then
		warningText = self.warningText[6];
	end;
	
	if self.warning[7] then
		g_currentMission:showBlinkingWarning(warningText, 3500);
	end;
	
	if self.showFlieglHelp then
		g_currentMission:addExtraPrintText(string.format(g_i18n:getText("FLIEGL_LOADING_TYPE"), autoLoadText));
		g_currentMission:addExtraPrintText(string.format(g_i18n:getText("FLIEGL_CHOOSEN_PRODUCTTYPE"), self.choosenProductType[self.productType][1]));
	
		if self.automaticLoading then
			if self.loadingIsActive then
				g_currentMission:addHelpButtonText(g_i18n:getText("FLIEGL_STOP_AUTOLOAD"), InputBinding.FLIEGL_LOAD);
				
				if self.fillLevel > 0 then
					g_currentMission:addExtraPrintText(g_i18n:getText("FLIEGL_STOP_AUTOLOAD_TO_UNLOAD"));
				end;
			else
				if self.fillLevel < self.capacity then
					g_currentMission:addHelpButtonText(g_i18n:getText("FLIEGL_START_AUTOLOAD"), InputBinding.FLIEGL_LOAD);
					g_currentMission:addHelpButtonText(g_i18n:getText("FLIEGL_TOGGLE_LOADING_TYPE"), InputBinding.FLIEGL_TOGGLE_LOADING_TYPE);
				end;
			end;
		else
			if self.fillLevel < self.capacity then
				g_currentMission:addHelpButtonText(g_i18n:getText("FLIEGL_LOAD"), InputBinding.FLIEGL_LOAD);
				g_currentMission:addHelpButtonText(g_i18n:getText("FLIEGL_TOGGLE_LOADING_TYPE"), InputBinding.FLIEGL_TOGGLE_LOADING_TYPE);
			end;
		end;
		
		g_currentMission:addHelpButtonText(g_i18n:getText("FLIEGL_UNLOAD"), InputBinding.FLIEGL_UNLOAD);
		g_currentMission:addHelpButtonText(g_i18n:getText("FLIEGL_TOGGLE_UNLOAD_SIDE"), InputBinding.FLIEGL_TOGGLE_UNLOAD_SIDE);
		g_currentMission:addExtraPrintText(string.format(g_i18n:getText("FLIEGL_UNLOAD_MODE"), self.currentUnloadSide[self.unloadSide][1]));
		
		if self.fillLevel > 0 and not self.loadingIsActive then	
			if self.showUnloadingHelp then
				g_currentMission:addHelpButtonText(g_i18n:getText("FLIEGL_HIDE_UNLOADING_HELP"), InputBinding.FLIEGL_TOGGLE_UNLOAD_HELP);
				
				local xStart, yStart, zStart;
				local xEnd, yEnd, zEnd;
				
				local xStart2, yStart2, zStart2;
				local xEnd2, yEnd2, zEnd2;
				
				local xStart3, yStart3, zStart3;
				local xEnd3, yEnd3, zEnd3;
				
				local xStart4, yStart4, zStart4;
				local xEnd4, yEnd4, zEnd4;
				
				if self.currentUnloadSide[self.unloadSide][1] == g_i18n:getText("UNLOAD_LEFT") then
					--## help line's left
					xStart, yStart, zStart = getWorldTranslation(self.helpLineLeftStart);
					xEnd, yEnd, zEnd = getWorldTranslation(self.helpLineLeftEnd);
					
					xStart2, yStart2, zStart2 = getWorldTranslation(self.helpLineLeftStart2);
					xEnd2, yEnd2, zEnd2 = getWorldTranslation(self.helpLineLeftEnd2);
					
					xStart3, yStart3, zStart3 = getWorldTranslation(self.helpLineLeftStart);
					xEnd3, yEnd3, zEnd3 = getWorldTranslation(self.helpLineLeftStart2);
					
					xStart4, yStart4, zStart4 = getWorldTranslation(self.helpLineLeftEnd);
					xEnd4, yEnd4, zEnd4 = getWorldTranslation(self.helpLineLeftEnd2);
				
				elseif self.currentUnloadSide[self.unloadSide][1] == g_i18n:getText("UNLOAD_RIGHT") then
					--## help line's right
					xStart, yStart, zStart = getWorldTranslation(self.helpLineRightStart);
					xEnd, yEnd, zEnd = getWorldTranslation(self.helpLineRightEnd);
					
					xStart2, yStart2, zStart2 = getWorldTranslation(self.helpLineRightStart2);
					xEnd2, yEnd2, zEnd2 = getWorldTranslation(self.helpLineRightEnd2);
					
					xStart3, yStart3, zStart3 = getWorldTranslation(self.helpLineRightStart);
					xEnd3, yEnd3, zEnd3 = getWorldTranslation(self.helpLineRightStart2);
					
					xStart4, yStart4, zStart4 = getWorldTranslation(self.helpLineRightEnd);
					xEnd4, yEnd4, zEnd4 = getWorldTranslation(self.helpLineRightEnd2);
				
				elseif self.currentUnloadSide[self.unloadSide][1] == g_i18n:getText("UNLOAD_BACK") then
					--## help line's back
					xStart, yStart, zStart = getWorldTranslation(self.helpLineBackStart);
					xEnd, yEnd, zEnd = getWorldTranslation(self.helpLineBackEnd);
					
					xStart2, yStart2, zStart2 = getWorldTranslation(self.helpLineBackStart2);
					xEnd2, yEnd2, zEnd2 = getWorldTranslation(self.helpLineBackEnd2);
					
					xStart3, yStart3, zStart3 = getWorldTranslation(self.helpLineBackStart);
					xEnd3, yEnd3, zEnd3 = getWorldTranslation(self.helpLineBackStart2);
					
					xStart4, yStart4, zStart4 = getWorldTranslation(self.helpLineBackEnd);
					xEnd4, yEnd4, zEnd4 = getWorldTranslation(self.helpLineBackEnd2);
				end;
				
				--## render help line's
				drawDebugLine(xStart, yStart, zStart, self.colorRed, self.colorGreen, self.colorBlue, xEnd, yEnd, zEnd, self.colorRed, self.colorGreen, self.colorBlue);
				drawDebugLine(xStart2, yStart2, zStart2, self.colorRed, self.colorGreen, self.colorBlue, xEnd2, yEnd2, zEnd2, self.colorRed, self.colorGreen, self.colorBlue);
				
				drawDebugLine(xStart3, yStart3, zStart3, self.colorRed, self.colorGreen, self.colorBlue, xEnd3, yEnd3, zEnd3, self.colorRed, self.colorGreen, self.colorBlue);
				drawDebugLine(xStart4, yStart4, zStart4, self.colorRed, self.colorGreen, self.colorBlue, xEnd4, yEnd4, zEnd4, self.colorRed, self.colorGreen, self.colorBlue);
			else
				g_currentMission:addHelpButtonText(g_i18n:getText("FLIEGL_SHOW_UNLOADING_HELP"), InputBinding.FLIEGL_TOGGLE_UNLOAD_HELP);
			end;
		end;
		
		if self.fillLevel == 0 then
			g_currentMission:addHelpButtonText(g_i18n:getText("FLIEGL_TOGGLE_PRODUCT_TYPE_UP"), InputBinding.FLIEGL_TOGGLE_PRODUCT_TYPE_UP);
			g_currentMission:addHelpButtonText(g_i18n:getText("FLIEGL_TOGGLE_PRODUCT_TYPE_DOWN"), InputBinding.FLIEGL_TOGGLE_PRODUCT_TYPE_DOWN);
			
			if self.attacherVehicle ~= nil and self.attacherVehicle:getIsActive() and self.attacherVehicle.productType ~= nil and self.attacherVehicle.productType ~= self.productType then
				g_currentMission:addHelpButtonText(g_i18n:getText("FLIEGL_TOGGLE_PRODUCT_FROM_FRONT_ATTACHED_TRAILER"), InputBinding.FLIEGL_TOGGLE_PRODUCT_FROM_FRONT_ATTACHED_TRAILER);
			end;
			
			for _, attachedTrailer in pairs(self.attachedImplements) do
				if attachedTrailer.object ~= nil then
					local backAttachedTrailer = attachedTrailer.object;
					
					if backAttachedTrailer:getIsActive() and backAttachedTrailer.productType ~= nil and backAttachedTrailer.productType ~= self.productType then
						g_currentMission:addHelpButtonText(g_i18n:getText("FLIEGL_TOGGLE_PRODUCT_FROM_BACK_ATTACHED_TRAILER"), InputBinding.FLIEGL_TOGGLE_PRODUCT_FROM_BACK_ATTACHED_TRAILER);
					end;
				end;
			end;
		end;
		
		g_currentMission:addHelpButtonText(g_i18n:getText("FLIEGL_HIDE_HELP"), InputBinding.FLIEGL_TOGGLE_HELP);
		
		if self.testMode then
			g_currentMission:addExtraPrintText(string.format(g_i18n:getText("FLIEGL_NUM_PRODUCTTYPES"), self.productType, #(self.numProducts)));
		end;
	else
		g_currentMission:addHelpButtonText(g_i18n:getText("FLIEGL_SHOW_HELP"), InputBinding.FLIEGL_TOGGLE_HELP);
	end;
end;

function FlieglMain:ProductInRangeFlieglMain(self, k)
	local nearestObject;
	local index;
	local nearestDistance = self.nearestDistance;
	local objectCopy = self.attacher[self.productType][k].object;
	local px, py, pz = getWorldTranslation(objectCopy);
	
	for index, vehicle in pairs(g_currentMission.vehicles) do
		local vehicleNode = vehicle.rootNode;
	
		if getParent(vehicleNode) == getRootNode() then
			local vx, vy, vz = getWorldTranslation(vehicleNode);
			local distance = Utils.vector3Length(px-vx, py-vy, pz-vz);
		
			if distance < nearestDistance then
				--[[testing something.. Maybe for future, we will see ..
				
				local vehicleFilename = vehicle.i3dFilename;
				
				if vehicleFilename ~= nil then
					print("Vehicle filename = '" .. tostring(vehicleFilename) .. "'.");
				else
					print("Vehicle filename = 'nil'.");
				end;]]
				
				local vehicleTypeName = vehicle.typeName;
				local vehicleAttacherJoint = vehicle.attacherJoint;
				local allowGrab = false;
				
				if self.choosenProductType[self.productType][1] == g_i18n:getText("TRANSPORTBOX") then
					if string.find(vehicleTypeName, "ROSbox") then
						if vehicleAttacherJoint == nil then --## do not steal a box from an frontloader or tractor, when it is attached on it ..
							allowGrab = true;
						end;
					end;
				elseif self.choosenProductType[self.productType][1] == g_i18n:getText("SILAGE_ADDITIVES_TANK") then
					if string.find(vehicleTypeName, "SilageAdditivesTank") then
						allowGrab = true;
					end;
				elseif self.choosenProductType[self.productType][1] == g_i18n:getText("BUYABLE_BLADES") then
					if string.find(vehicleTypeName, "BuyableBlades") then
						allowGrab = true;
					end;
				elseif self.choosenProductType[self.productType][1] == g_i18n:getText("BM_MODDING_PALLETS") then
					if string.find(vehicleTypeName, "netRolePallet") or string.find(vehicleTypeName, "PalletFoilCartons") then
						allowGrab = true;
					end;
				elseif self.choosenProductType[self.productType][1] == g_i18n:getText("CONSUMPTION_GOODS") then
					if string.find(vehicleTypeName, "BuyableBlades") 
						or string.find(vehicleTypeName, "SilageAdditivesTank")
						or string.find(vehicleTypeName, "netRolePallet")
						or string.find(vehicleTypeName, "PalletFoilCartons")
					then
						allowGrab = true;
					end;
				end;
				
				if allowGrab then
					index = index;
					nearestObject = vehicleNode;
					nearestDistance = distance;
				end;
			end;
		end;
	end;
	
	for index, item in pairs(g_currentMission.itemsToSave) do
		local itemNode = item.item.nodeId;
		
		if getParent(itemNode) == getRootNode() then
			local vx, vy, vz = getWorldTranslation(itemNode);
			local distance = Utils.vector3Length(px-vx, py-vy, pz-vz);
			
			if distance < nearestDistance then
				local fillType = Utils.getNoNil(getUserAttribute(itemNode, "fillType"), "wheat_windrow");
				local capacity = item.item.capacity;
				local fillLevel = item.item.fillLevel;
				local isRoundbale = Utils.getNoNil(getUserAttribute(itemNode, "isRoundbale"), false);
				--local isSquarebale = Utils.getNoNil(getUserAttribute(itemNode, "isSquarebale"), false);
				local wrappingState = Utils.getNoNil(item.item.wrappingState, 0);
				local filename = item.item.i3dFilename;
				local isTreeSaplingPallet = Utils.getNoNil(getUserAttribute(itemNode, "deleteIfEmpty"), false);
				local startFillLevel = Utils.getNoNil(getUserAttribute(itemNode, "startFillLevel"), 0);
				local allowGrab = false;
				
				--## fix typing mistake by GIANTS in the UserAttribute 'fillType' from the straw roundbales ..
				if fillType == "wheat_wiindrow" then
					fillType = "wheat_windrow";
				end;
				
				if self.choosenProductType[self.productType][1] == g_i18n:getText("STRAW_SQUAREBALE") then
					if string.find(filename, "baleStraw240") then
						allowGrab = true;
					end;
				elseif self.choosenProductType[self.productType][1] == g_i18n:getText("HAY_SQUAREBALE") then
					if string.find(filename, "baleHay240") then
						allowGrab = true;
					end;
				elseif self.choosenProductType[self.productType][1] == g_i18n:getText("MIX_SQUAREBALE") then
					if string.find(filename, "baleHay240") or string.find(filename, "baleStraw240") then
						allowGrab = true;
					end;
				elseif self.choosenProductType[self.productType][1] == g_i18n:getText("STANDARD_STRAW_ROUNDBALE") then
					if isRoundbale 
						and not string.find(filename, "Bale130Straw")
						and not string.find(filename, "Bale140Straw") 
						and not string.find(filename, "Bale150Straw")
					then
						if fillType == "wheat_windrow" then
							allowGrab = true;
						end;
					end;
				elseif self.choosenProductType[self.productType][1] == g_i18n:getText("STANDARD_HAY_ROUNDBALE") then
					if isRoundbale 
						and not string.find(filename, "Bale130Hay") 
						and not string.find(filename, "Bale140Hay")
						and not string.find(filename, "Bale150Hay")
					then
						if fillType == "dryGrass_windrow" and wrappingState == 0 then
							allowGrab = true;
						end;
					end;
				elseif self.choosenProductType[self.productType][1] == g_i18n:getText("STANDARD_GRASS_ROUNDBALE") then
					if isRoundbale 
						and not string.find(filename, "Bale130Grass") 
						and not string.find(filename, "Bale140Grass") 
						and not string.find(filename, "Bale150Grass")
					then
						if fillType == "grass_windrow" and wrappingState == 0 then
							allowGrab = true;
						end;
					end;
				elseif self.choosenProductType[self.productType][1] == g_i18n:getText("STANDARD_SILAGE_ROUNDBALE") then
					if isRoundbale and wrappingState == 1 then
						allowGrab = true;
					end;
				elseif self.choosenProductType[self.productType][1] == g_i18n:getText("STANDARD_MIX_ROUNDBALE") then
					if isRoundbale 
						and not string.find(filename, "Bale130Straw")
						and not string.find(filename, "Bale140Straw")
						and not string.find(filename, "Bale150Straw")
						and not string.find(filename, "Bale130Grass")
						and not string.find(filename, "Bale140Grass")
						and not string.find(filename, "Bale150Grass")
						and not string.find(filename, "Bale130Silage") 
						and not string.find(filename, "Bale140Silage") 
						and not string.find(filename, "Bale150Silage") 
						and not string.find(filename, "Bale130Hay") 
						and not string.find(filename, "Bale140Hay") 
						and not string.find(filename, "Bale150Hay")
					then
						allowGrab = true;
					end;
				
				--## BM Modding support
				elseif self.choosenProductType[self.productType][1] == g_i18n:getText("BM_MODDING_STRAW_ROUNDBALE_130") then
					if string.find(filename, "Bale130Straw") then
						allowGrab = true;
					end;
				elseif self.choosenProductType[self.productType][1] == g_i18n:getText("BM_MODDING_HAY_ROUNDBALE_130") then
					if string.find(filename, "Bale130Hay") then
						allowGrab = true;
					end;
				elseif self.choosenProductType[self.productType][1] == g_i18n:getText("BM_MODDING_GRASS_ROUNDBALE_130") then
					if string.find(filename, "Bale130Grass") then
						allowGrab = true;
					end;
				elseif self.choosenProductType[self.productType][1] == g_i18n:getText("BM_MODDING_SILAGE_ROUNDBALE_130") then
					if string.find(filename, "Bale130Silage") then
						allowGrab = true;
					end;
				elseif self.choosenProductType[self.productType][1] == g_i18n:getText("BM_MODDING_MIX_ROUNDBALE_130") then
					if string.find(filename, "Bale130Hay") 
						or string.find(filename, "Bale130Grass")
						or string.find(filename, "Bale130Silage")
						or string.find(filename, "Bale130Straw")
					then
						allowGrab = true;
					end;
				elseif self.choosenProductType[self.productType][1] == g_i18n:getText("BM_MODDING_STRAW_ROUNDBALE_140") then
					if string.find(filename, "Bale140Straw") then
						allowGrab = true;
					end;
				elseif self.choosenProductType[self.productType][1] == g_i18n:getText("BM_MODDING_HAY_ROUNDBALE_140") then
					if string.find(filename, "Bale140Hay") then
						allowGrab = true;
					end;
				elseif self.choosenProductType[self.productType][1] == g_i18n:getText("BM_MODDING_GRASS_ROUNDBALE_140") then
					if string.find(filename, "Bale140Grass") then
						allowGrab = true;
					end;
				elseif self.choosenProductType[self.productType][1] == g_i18n:getText("BM_MODDING_SILAGE_ROUNDBALE_140") then
					if string.find(filename, "Bale140Silage") then
						allowGrab = true;
					end;
				elseif self.choosenProductType[self.productType][1] == g_i18n:getText("BM_MODDING_MIX_ROUNDBALE_140") then
					if string.find(filename, "Bale140Hay") 
						or string.find(filename, "Bale140Grass")
						or string.find(filename, "Bale140Silage")
						or string.find(filename, "Bale140Straw")
					then
						allowGrab = true;
					end;
				elseif self.choosenProductType[self.productType][1] == g_i18n:getText("BM_MODDING_STRAW_ROUNDBALE_150") then
					if string.find(filename, "Bale150Straw") then
						allowGrab = true;
					end;
				elseif self.choosenProductType[self.productType][1] == g_i18n:getText("BM_MODDING_HAY_ROUNDBALE_150") then
					if string.find(filename, "Bale150Hay") then
						allowGrab = true;
					end;
				elseif self.choosenProductType[self.productType][1] == g_i18n:getText("BM_MODDING_GRASS_ROUNDBALE_150") then
					if string.find(filename, "Bale150Grass") then
						allowGrab = true;
					end;
				elseif self.choosenProductType[self.productType][1] == g_i18n:getText("BM_MODDING_SILAGE_ROUNDBALE_150") then
					if string.find(filename, "Bale150Silage") then
						allowGrab = true;
					end;
				elseif self.choosenProductType[self.productType][1] == g_i18n:getText("BM_MODDING_MIX_ROUNDBALE_150") then
					if string.find(filename, "Bale150Hay") 
						or string.find(filename, "Bale150Grass")
						or string.find(filename, "Bale150Silage")
						or string.find(filename, "Bale150Straw")
					then
						allowGrab = true;
					end;
				elseif self.choosenProductType[self.productType][1] == g_i18n:getText("WOOLPALLET") then
					if string.find(filename, "woolPallet") then
						allowGrab = true;
					end;
				elseif self.choosenProductType[self.productType][1] == g_i18n:getText("STANDARD_TREESAPLINGPALLET") then
					if isTreeSaplingPallet and (startFillLevel == 20 or startFillLevel == 40 or startFillLevel == 120) then
						allowGrab = true;
					end;
				elseif self.choosenProductType[self.productType][1] == g_i18n:getText("BIG_TREESAPLINGPALLET") then
					if isTreeSaplingPallet and startFillLevel == 196 then
						allowGrab = true;
					end;
				elseif self.choosenProductType[self.productType][1] == g_i18n:getText("BOARDPALLET") then 
					--## support for marhu's board pallets
					if fillType == "woodChips" and fillLevel == capacity then
						allowGrab = true;
					end;
				elseif self.choosenProductType[self.productType][1] == g_i18n:getText("WASHEDPOTATOPALLET") then 
					--## support for Farmer_Andy's washed potato pallets
					if fillType == "washedPotato" then
						allowGrab = true;
					end;
				elseif self.choosenProductType[self.productType][1] == g_i18n:getText("EGGPALLET") then 
					--## support for Farmer_Andy's egg pallets
					if fillType == "egg" then
						allowGrab = true;
					end;
				elseif self.choosenProductType[self.productType][1] == g_i18n:getText("GREENHOUSEPALLET") then 
					--## support for greenhouse pallets
					if fillType == "raspberry" 
						or fillType == "strawberry"
						or fillType == "blumenkohl"
						or fillType == "tomate"
						or fillType == "lettuce"
						or fillType == "kirsche"
						or fillType == "pflaume"
					then
						allowGrab = true;
					end;
				elseif self.choosenProductType[self.productType][1] == g_i18n:getText("H_MILK") then 
					--## support for Farmer_Andy's H-milk cans
					if fillType == "hmilk" then
						allowGrab = true;
					end;
				end;
				
				if allowGrab then
					index = index;
					nearestObject = itemNode;
					nearestDistance = distance;
				end;
			end;
		end;
	end;

	return nearestObject;
end;

function FlieglMain:ProductAttachFlieglMain(self, k)
	local x, y, z = getWorldRotation(self.attacher[self.productType][k].productInRange);
	
	setTranslation(self.attacher[self.productType][k].productInRange, 0, 0, 0);
	setRotation(self.attacher[self.productType][k].productInRange, 0, 0, 0);
	removeFromPhysics(self.attacher[self.productType][k].productInRange);
	link(self.attacher[self.productType][k].object, self.attacher[self.productType][k].productInRange);
	self.attacher[self.productType][k].attachedProduct = self.attacher[self.productType][k].productInRange;
	self.attacher[self.productType][k].productInRange = nil;
	self.fillLevel = self.fillLevel + 1;
end;

function FlieglMain:ProductDetachFlieglMain(self, k)
	if self.attacher[self.productType][k].attachedProduct ~= nil then
		if self.currentUnloadSide[self.unloadSide][1] == g_i18n:getText("UNLOAD_TRAILER") then
			local x, y, z = getWorldTranslation(self.attacher[self.productType][k].attachedProduct);
			local rx, ry, rz = getWorldRotation(self.attacher[self.productType][k].attachedProduct);
			local root = getRootNode();
			
			setTranslation(self.attacher[self.productType][k].attachedProduct, x, y, z);
			setRotation(self.attacher[self.productType][k].attachedProduct, rx, ry, rz);
			addToPhysics(self.attacher[self.productType][k].attachedProduct);
			link(root, self.attacher[self.productType][k].attachedProduct);
			self.attacher[self.productType][k].attachedProduct = nil;
		else
			local x, y, z = getWorldTranslation(self.attacher[self.productType][k].attachedProduct);
			local rx, ry, rz = getWorldRotation(self.attacher[self.productType][k].attachedProduct);
			local nx, ny, nz = getWorldTranslation(self.attacherLevel[self.productType]);
			local tx, ty, tz = getWorldTranslation(self.currentUnloadSide[self.unloadSide][3]);
			local x = x + (tx - nx);
			local y = y + (ty - ny);
			local z = z + (tz - nz);
			local tH = getTerrainHeightAtWorldPos(g_currentMission.terrainRootNode, x, y, z);
			local relHeight = ny - tH;
			local root = getRootNode();
			
			setTranslation(self.attacher[self.productType][k].attachedProduct, x, (y - relHeight), z);
			setRotation(self.attacher[self.productType][k].attachedProduct, rx, ry, rz);
			addToPhysics(self.attacher[self.productType][k].attachedProduct);
			link(root, self.attacher[self.productType][k].attachedProduct);
			self.attacher[self.productType][k].attachedProduct = nil;
		end;
	end;
end;

function FlieglMain:onDetach()
	if self.loadingIsActive then
		self.loadingIsActive = false;
	end;
end;

function FlieglMain:onLeave()
	if self.loadingIsActive then
		self.loadingIsActive = false;
	end;
end;