--
-- ThreshingCounter
-- Specialization for counting threshed hectars
--
-- @author:  	Manuel Leithner
-- @date:		23/10/12
-- @version:	v2.0
-- @history:	v2.0 - initial implementation
--
-- free for noncommerical-usage
--
-- Geändert am: 28.11.2014
-- Geändert von: Ifko[nator]
-- Geädert wurde: Konvert in den LS 15 und als globalen Mod erstellt.

ThreshingCounter = {};

local ThreshingCounterDirectory = g_currentModDirectory;

function ThreshingCounter.prerequisitesPresent(specializations)
    return SpecializationUtil.hasSpecialization(Combine, specializations);
end;

function ThreshingCounter:load(xmlFile)
	self.sessionHectars = 0;	
	self.totalHectars = 0;
	
	self.threshingCounterHudPosX = 0.7997;
	self.threshingCounterHudPosY = 0.1679;	
	
	self.threshingCounterOverlay = Overlay:new("threshingCounterOverlay", Utils.getFilename("ThreshingCounter/ThreshingCounter_hud.png", ThreshingCounterDirectory), self.threshingCounterHudPosX, self.threshingCounterHudPosY, 0.2371, 0.039525);	
	self.threshingCounterOverlayTransparent = Overlay:new("threshingCounterOverlayTransparent", Utils.getFilename("ThreshingCounter/ThreshingCounter_Transparent_hud.png", ThreshingCounterDirectory), self.threshingCounterHudPosX, self.threshingCounterHudPosY, 0.2371, 0.039525);	
	
	self.showTransparentHud = true;
	
	self.sessionHectarsSent = 0;
	self.totalHectarsSent = 0;
	self.threshingCounterDirtyFlag = self:getNextDirtyFlag();
end;

function ThreshingCounter:delete()
end;

function ThreshingCounter:readStream(streamId, connection)
	local session = streamReadFloat32(streamId);
	local total = streamReadFloat32(streamId);	
	
	self.sessionHectars = session;
	self.totalHectars = total;
	
	self.sessionHectarsSent = self.sessionHectars;
	self.totalHectarsSent = self.totalHectars;
end;

function ThreshingCounter:writeStream(streamId, connection)	
	streamWriteFloat32(streamId, self.sessionHectars);
	streamWriteFloat32(streamId, self.totalHectars);
end;

function ThreshingCounter:readUpdateStream(streamId, timestamp, connection)
    if connection:getIsServer() then
        if streamReadBool(streamId) then
			self.sessionHectars = streamReadFloat32(streamId);
			self.totalHectars = streamReadFloat32(streamId);
        end;
    end;
end;

function ThreshingCounter:writeUpdateStream(streamId, connection, dirtyMask)
    if not connection:getIsServer() then
        if streamWriteBool(streamId, bitAND(dirtyMask, self.threshingCounterDirtyFlag) ~= 0) then
			streamWriteFloat32(streamId, self.sessionHectarsSent);
			streamWriteFloat32(streamId, self.totalHectarsSent);
        end;
    end;
end;

function ThreshingCounter:mouseEvent(posX, posY, isDown, isUp, button)
end;

function ThreshingCounter:keyEvent(unicode, sym, modifier, isDown)
end;

function ThreshingCounter:update(dt)
	if self:getIsActiveForInput() then
		if InputBinding.hasEvent(InputBinding.SHOW_TRANSPARENT_HUD_BUTTON) then
			self.showTransparentHud = not self.showTransparentHud;
		end;
	end;
end;

function ThreshingCounter:updateTick(dt)
end;

function ThreshingCounter:draw()
	local counterSession = math.floor(self.sessionHectars * 100) / 100;
	local counterTotal = math.floor(self.totalHectars * 10) / 10;
	
	local fullSession = math.floor(counterSession);
	local fullTotal = math.floor(counterTotal);
	
	local deciSession = math.floor((counterSession - fullSession) * 100);
	local deciTotal = math.floor((counterTotal - fullTotal) * 10);
	
	if deciSession < 10 then
		deciSession = "0" .. deciSession;
	end;
	
	if self.showTransparentHud then
		self.threshingCounterOverlayTransparent:render();
		if self:getIsActiveForInput() then
			g_currentMission:addHelpButtonText(Combine.SHOW_NOT_TRANSPARENT_HUD_TEXT, InputBinding.SHOW_TRANSPARENT_HUD_BUTTON);
		end;
	else
		self.threshingCounterOverlay:render();
		if self:getIsActiveForInput() then
			g_currentMission:addHelpButtonText(Combine.SHOW_TRANSPARENT_HUD_TEXT, InputBinding.SHOW_TRANSPARENT_HUD_BUTTON);
		end;
	end;
	
	setTextAlignment(RenderText.ALIGN_RIGHT);
	setTextBold(true);	
	
	setTextColor(0, 0, 0, 1); -- Black
	renderText(self.threshingCounterHudPosX + 0.089, 			  self.threshingCounterHudPosY + 0.009, 0.018, tostring(fullTotal) .. ".");
	renderText(self.threshingCounterHudPosX + 0.05 + 0.2371 / 2,  self.threshingCounterHudPosY + 0.009, 0.018, tostring(fullSession) .. ".");
	renderText(self.threshingCounterHudPosX + 0.095, 			  self.threshingCounterHudPosY + 0.009, 0.019, tostring(deciTotal));
	renderText(self.threshingCounterHudPosX + 0.062 + 0.2371 / 2, self.threshingCounterHudPosY + 0.009, 0.019, tostring(deciSession));
	renderText(self.threshingCounterHudPosX + 0.11, 			  self.threshingCounterHudPosY + 0.009, 0.019, "ha");
	renderText(self.threshingCounterHudPosX + 0.195, 			  self.threshingCounterHudPosY + 0.009, 0.019, "ha");
	
	setTextColor(1, 1, 1, 1); -- Wihte
	renderText(self.threshingCounterHudPosX + 0.089, 			  self.threshingCounterHudPosY + 0.011, 0.018, tostring(fullTotal) .. ".");
	renderText(self.threshingCounterHudPosX + 0.05 + 0.2371 / 2,  self.threshingCounterHudPosY + 0.011, 0.018, tostring(fullSession) .. ".");
	renderText(self.threshingCounterHudPosX + 0.11, 			  self.threshingCounterHudPosY + 0.011, 0.019, "ha");
	renderText(self.threshingCounterHudPosX + 0.195, 			  self.threshingCounterHudPosY + 0.011, 0.019, "ha");
	
	setTextColor(0.95, 0, 0, 1); -- Red
	renderText(self.threshingCounterHudPosX + 0.095, 			  self.threshingCounterHudPosY + 0.011, 0.019, tostring(deciTotal));
	renderText(self.threshingCounterHudPosX + 0.062 + 0.2371 / 2, self.threshingCounterHudPosY + 0.011, 0.019, tostring(deciSession));
	
	setTextAlignment(RenderText.ALIGN_LEFT);
	setTextBold(false);	
end;

function ThreshingCounter:loadFromAttributesAndNodes(xmlFile, key, resetVehicles)
	if not resetVehicles then
		local totalHectars = Utils.getNoNil(getXMLFloat(xmlFile, key .. "#totalHectars"), self.totalHectars);
		local showTransparentHud = Utils.getNoNil(getXMLBool(xmlFile, key .. "#showTransparentHud"), self.showTransparentHud);
		
		self.totalHectars = totalHectars;
		self.showTransparentHud = showTransparentHud;
	end;
	
    return BaseMission.VEHICLE_LOAD_OK;
end;

function ThreshingCounter:getSaveAttributesAndNodes(nodeIdent)
	local attributes = 
		'totalHectars="' .. tostring(self.totalHectars) .. 
		'" showTransparentHud="' .. tostring(self.showTransparentHud) ..
	'"';
	
	return attributes, nil;
end;

function ThreshingCounter:addCutterArea(cutter, area, realArea, inputFruitType, fruitType)
	if area > 0 and (self.lastCuttersFruitType == FruitUtil.FRUITTYPE_UNKNOWN or self.lastCuttersArea == 0 or self.lastCuttersFruitType == fruitType) then
		local pixelToSqm = g_currentMission:getFruitPixelsToSqm();
		local sqm = realArea * pixelToSqm;
		local ha = sqm / 10000;			
		
		self.sessionHectars = self.sessionHectars + ha;		
		self.totalHectars = self.totalHectars + ha;	
		
		if math.abs(self.sessionHectars - self.sessionHectarsSent) > 0.01 or math.abs(self.totalHectars - self.totalHectarsSent) > 0.01 then
            self:raiseDirtyFlags(self.threshingCounterDirtyFlag);
			self.sessionHectarsSent = self.sessionHectars;
			self.totalHectarsSent = self.totalHectars;
        end;
	end;
end;