--
-- BuyableFergieWheels
-- Specialization for Buyable Twin Wheels
--
-- @author  	Manuel Leithner (SFM-Modding)
-- @version 	v3.0
-- @date  		30/10/10
-- @history:	v1.0 - Initial version
--				v2.0 - added network support, changed update to updateTick
--				v3.0 - Added dynamic collision support, ls13-ready
--				v3.1 - Some modifications (by fruktor) 
--
-- free for noncommerical-usage
--

BuyableFergieWheels = {};

function BuyableFergieWheels.prerequisitesPresent(specializations)
    return true;
end;

function BuyableFergieWheels:load(xmlFile)

	self.twinWheelTriggerCallback = BuyableFergieWheels.twinWheelTriggerCallback;
	self.wheelDelete = BuyableFergieWheels.wheelDelete;
	self.assembleTwinWheels = SpecializationUtil.callSpecializationsFunction("assembleTwinWheels");
	self.disassembleTwinWheels = SpecializationUtil.callSpecializationsFunction("disassembleTwinWheels");
	self.assembleCageWheels = SpecializationUtil.callSpecializationsFunction("assembleCageWheels");
	self.disassembleCageWheels = SpecializationUtil.callSpecializationsFunction("disassembleCageWheels");

	self.checkString = Utils.getNoNil(getXMLString(xmlFile, "vehicle.extraWheels#checkString"), "standard");
	self.cageCheckString = Utils.getNoNil(getXMLString(xmlFile, "vehicle.extraWheels#cageWheelCheckString"), "standard");
	self.usingTrigger = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.extraWheels#activationTrigger"));
	
	addTrigger(self.usingTrigger, "twinWheelTriggerCallback", self);
    self.deleteListenerId = addDeleteListener(self.usingTrigger, "wheelDelete", self);
	
	self.twinBelts = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.extraWheels#twinBelts"));
	
    self.twinWheels = {};
	local i = 0;
    while true do
        local wheelnamei = string.format("vehicle.extraWheels.twinWheel(%d)", i);
        local wheel = {};
        local wheelIndex = getXMLInt(xmlFile, wheelnamei .. "#wheelIndex");
        if wheelIndex == nil then
            break;
        end;
		
		wheel.wheelIndex = wheelIndex;
		wheel.node = Utils.indexToObject(self.components, getXMLString(xmlFile, wheelnamei .. "#node"));
		wheel.savePosition = Utils.indexToObject(self.components, getXMLString(xmlFile, wheelnamei .. "#savePosition"));
		table.insert(self.twinWheels, wheel);
		i = i + 1;
	end;
	
	self.cageBelts = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.extraWheels#cageBelts"));
	self.cageBeltsOff = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.extraWheels#cageBeltsOff"));

    self.cageWheels = {};
    local i = 0;
    while true do
        local wheelnamei = string.format("vehicle.extraWheels.cageWheel(%d)", i);
        local wheel = {};
        local wheelIndex = getXMLInt(xmlFile, wheelnamei .. "#wheelIndex");
        if wheelIndex == nil then
            break;
        end;
		
		wheel.wheelIndex = wheelIndex;
		wheel.node = Utils.indexToObject(self.components, getXMLString(xmlFile, wheelnamei .. "#node"));
		wheel.savePosition = Utils.indexToObject(self.components, getXMLString(xmlFile, wheelnamei .. "#savePosition"));
		table.insert(self.cageWheels, wheel);
		i = i + 1;
	end;

	self.loadedTwinCoords = nil;
	self.loadedCageCoords = nil;
	self.twinWheelsUser = nil;
	self.cageWheelsUser = nil;
end;

function BuyableFergieWheels:delete()
	if self.twinWheelsUser ~= nil then
		self.twinWheelsUser:onDisassemblingTwins(true);
		for _,twinWheel in pairs(self.twinWheels) do
			delete(twinWheel.node);
		end;
		self.twinWheels = {};
	end;
	if self.cageWheelsUser ~= nil then
		self.cageWheelsUser:onDisassemblingCage(true);
		self:disassembleCageWheels();
		self.cageWheels = {};
	end;
    removeTrigger(self.usingTrigger);
end;

function BuyableFergieWheels:wheelDelete()
    removeDeleteListener(self.usingTrigger, self.deleteListenerId);
end;

function BuyableFergieWheels:readStream(streamId, connection)
	local id = streamReadInt32(streamId);
	if id ~= -1 then
		local twinWheelsUser = networkGetObject(id);
		if twinWheelsUser ~= nil and cageWheelsUser then
			self:assembleTwinWheels(twinWheelsUser, true);
		end;
	end;
	local id = streamReadInt32(streamId);
	if id ~= -1 then
		local cageWheelsUser = networkGetObject(id);
		if cageWheelsUser ~= nil and cageWheelsUser then
			self:assembleCageWheels(cageWheelsUser, true);
		end;
	end;
end;

function BuyableFergieWheels:writeStream(streamId, connection)
	local idToWrite = -1;
	if self.twinWheelsUser ~= nil then
		idToWrite = networkGetObjectId(self.twinWheelsUser);
	end;
	streamWriteInt32(streamId, idToWrite);
	local idToWrite = -1;
	if self.cageWheelsUser ~= nil then
		idToWrite = networkGetObjectId(self.cageWheelsUser);
	end;
	streamWriteInt32(streamId, idToWrite);
end;

function BuyableFergieWheels:mouseEvent(posX, posY, isDown, isUp, button)
end;

function BuyableFergieWheels:keyEvent(unicode, sym, modifier, isDown)
end;

function BuyableFergieWheels:loadFromAttributesAndNodes(xmlFile, key, resetVehicles)

	if not resetVehicles then
		local extraWheelsCoords = getXMLString(xmlFile, key.."#extraWheelsCoords");		
		if extraWheelsCoords ~= nil then
			local x1,y1,z1,x2,y2,z2 = Utils.getVectorFromString(extraWheelsCoords);
			if x1 ~= nil and y1 ~= nil and z1 ~= nil and x1 ~= 0 and y1 ~= 0 and z1 ~= 0 then 
				self.loadedTwinCoords = {x = x1, y = y1, z = z1};
			end;
			if x2 ~= nil and y2 ~= nil and z2 ~= nil and x2 ~= 0 and y2 ~= 0 and z2 ~= 0 then 
				self.loadedCageCoords = {x = x2, y = y2, z = z2};
			end;
		end;
	end; 

    return BaseMission.VEHICLE_LOAD_OK;
end;

function BuyableFergieWheels:getSaveAttributesAndNodes(nodeIdent)	
    local attributes = nil;
	local x1,y1,z1 = 0, 0, 0;
	local x2,y2,z2 = 0, 0, 0;
	if self.twinWheelsUser ~= nil then
		x1,y1,z1 = getWorldTranslation(self.twinWheelsUser.rootNode);
	end;
	if self.cageWheelsUser ~= nil then
		x2,y2,z2 = getWorldTranslation(self.cageWheelsUser.rootNode);
	end;
	attributes = 'extraWheelsCoords="'.. x1 .. " " .. y1 .. " " .. z1 .. " " .. x2 .. " " .. y2 .. " " .. z2 .. '"';

    return attributes, nil;
end;

function BuyableFergieWheels:update(dt)
end;

function BuyableFergieWheels:updateTick(dt)
	if self.loadedTwinCoords ~= nil then
		for k,steerable in pairs(g_currentMission.steerables) do
			local a,b,c = getWorldTranslation(steerable.rootNode);
			local distance = Utils.vector3Length(self.loadedTwinCoords.x-a, self.loadedTwinCoords.y-b, self.loadedTwinCoords.z-c);
			if distance < 0.15 then				
				self:assembleTwinWheels(steerable);
				break;
			end;
		end;
		self.loadedTwinCoords = nil;
	end;
	if self.loadedCageCoords ~= nil then
		for k,steerable in pairs(g_currentMission.steerables) do
			local a,b,c = getWorldTranslation(steerable.rootNode);
			local distance = Utils.vector3Length(self.loadedCageCoords.x-a, self.loadedCageCoords.y-b, self.loadedCageCoords.z-c);
			if distance < 0.15 then				
				self:assembleCageWheels(steerable);
				break;
			end;
		end;
		self.loadedCageCoords = nil;
	end;
end;

function BuyableFergieWheels:draw()	
end;

function BuyableFergieWheels:twinWheelTriggerCallback(triggerId, otherId, onEnter, onLeave, onStay, otherShapeId)
	local vehicle = g_currentMission.controlledVehicle;
	if vehicle ~= nil then
		if onEnter then		
			if vehicle.rootNode == otherId then
				if vehicle.buyableTwinWheels ~= nil then
					if vehicle.buyableTwinWheels.checkString == self.checkString then
						if self.twinWheelsUser ~= nil then
							if self.twinWheelsUser == vehicle then
								vehicle.buyableTwinWheels.wheelsInRange = self;
							end;
						else
							vehicle.buyableTwinWheels.wheelsInRange = self;
						end;
					end;
				end;
				if vehicle.buyableCageWheels ~= nil then
					if vehicle.buyableCageWheels.checkString == self.checkString or vehicle.buyableCageWheels.checkString == self.cageCheckString then
						if self.cageWheelsUser ~= nil then
							if self.cageWheelsUser == vehicle then
								vehicle.buyableCageWheels.wheelsInRange = self;
							end;
						else
							vehicle.buyableCageWheels.wheelsInRange = self;
						end;
					end;
				end;
			end;
		elseif onLeave then		
			if otherId == vehicle.rootNode then
				if vehicle.buyableTwinWheels ~= nil then
					vehicle.buyableTwinWheels.wheelsInRange = nil;
				end;
				if vehicle.buyableCageWheels ~= nil then
					vehicle.buyableCageWheels.wheelsInRange = nil;
				end;
			end;
		end;
	end;
end;

function BuyableFergieWheels:assembleTwinWheels(vehicle, noEventSend)
	BuyableMFdualsAttachEvent.sendEvent(self, vehicle, noEventSend);
	
	if self.twinBelts ~= nil then
		setVisibility(self.twinBelts, false);
	end;
	
	if self.twinWheelsUser == nil then
		if vehicle.buyableTwinWheels ~= nil then
			self.twinWheelsUser = vehicle;	
			for l,twinWheel in pairs(self.twinWheels) do 				
				for k,wheel in pairs(vehicle.wheels) do
					if k == twinWheel.wheelIndex then
						link(wheel.driveNode, twinWheel.node);
						local x,y,z = getWorldTranslation(wheel.driveNode);							
						x,y,z = worldToLocal(getParent(twinWheel.node), x,y,z);
						setTranslation(twinWheel.node,x,y,z);
						break;
					end;
				end;
			end;			
			self.twinWheelsUser:onAssemblingTwins(self);
		end;
	end;
end;

function BuyableFergieWheels:disassembleTwinWheels(noEventSend)
	BuyableMFdualsDetachEvent.sendEvent(self, noEventSend);
	
	if self.twinBelts ~= nil then
		setVisibility(self.twinBelts, true);		
	end;

	self.twinWheelsUser:onDisassemblingTwins();
	
	for k,wheel in pairs(self.twinWheels) do
		link(wheel.savePosition, wheel.node);
		setRotation(wheel.node, 0,0,0);
		setTranslation(wheel.node, 0,0,0);
	end;	
	self.twinWheelsUser = nil;
end;

function BuyableFergieWheels:assembleCageWheels(vehicle, noEventSend)
	BuyableMFcageAttachEvent.sendEvent(self, vehicle, noEventSend);
	
	if self.cageBelts ~= nil and self.cageBeltsOff ~= nil then
		setVisibility(self.cageBelts, false);
		setVisibility(self.cageBeltsOff, true);
	end;
	
	if self.cageWheelsUser == nil then
		if vehicle.buyableCageWheels ~= nil then
			self.cageWheelsUser = vehicle;	
			for l,cageWheel in pairs(self.cageWheels) do
				for k,wheel in pairs(vehicle.wheels) do
					if k == cageWheel.wheelIndex then
						link(wheel.driveNode, cageWheel.node);
						local x,y,z = getWorldTranslation(wheel.driveNode);							
						x,y,z = worldToLocal(getParent(cageWheel.node), x,y,z);
						setTranslation(cageWheel.node,x,y,z);
						break;
					end;
				end;
			end;			
			self.cageWheelsUser:onAssemblingCage(self);
		end;
	end;
end;

function BuyableFergieWheels:disassembleCageWheels(noEventSend)
	BuyableMFcageDetachEvent.sendEvent(self, noEventSend);
	
	if self.cageBelts ~= nil and self.cageBeltsOff ~= nil then
		setVisibility(self.cageBelts, true);
		setVisibility(self.cageBeltsOff, false);
	end;

	self.cageWheelsUser:onDisassemblingCage();
	
	for k,cageWheel in pairs(self.cageWheels) do
		link(cageWheel.savePosition, cageWheel.node);
		setRotation(cageWheel.node, 0,0,0);
		setTranslation(cageWheel.node, 0,0,0);
	end;	
	self.cageWheelsUser = nil;
end;


BuyableMFdualsAttachEvent = {};
BuyableMFdualsAttachEvent_mt = Class(BuyableMFdualsAttachEvent, Event);

InitEventClass(BuyableMFdualsAttachEvent, "BuyableMFdualsAttachEvent");

function BuyableMFdualsAttachEvent:emptyNew()
    local self = Event:new(BuyableMFdualsAttachEvent_mt);
    return self;
end;

function BuyableMFdualsAttachEvent:new(vehicle, attacherVehicle)
    local self = BuyableMFdualsAttachEvent:emptyNew()
    self.vehicle = vehicle;
	self.attacherVehicle = attacherVehicle;
    return self;
end;

function BuyableMFdualsAttachEvent:readStream(streamId, connection)
    local id = streamReadInt32(streamId);
	local attacherId = streamReadInt32(streamId);
	self.attacherVehicle = networkGetObject(attacherId);
    self.vehicle = networkGetObject(id);
    self:run(connection);
end;

function BuyableMFdualsAttachEvent:writeStream(streamId, connection)
    streamWriteInt32(streamId, networkGetObjectId(self.vehicle));
	streamWriteInt32(streamId, networkGetObjectId(self.attacherVehicle));
end;

function BuyableMFdualsAttachEvent:run(connection)
	self.vehicle:assembleTwinWheels(self.attacherVehicle, true);
    if not connection:getIsServer() then
        g_server:broadcastEvent(BuyableMFdualsAttachEvent:new(self.vehicle, self.attacherVehicle), nil, connection, self.object);
    end;
end;


function BuyableMFdualsAttachEvent.sendEvent(vehicle, attacherVehicle, noEventSend)
	if noEventSend == nil or noEventSend == false then
		if g_server ~= nil then
			g_server:broadcastEvent(BuyableMFdualsAttachEvent:new(vehicle, attacherVehicle), nil, nil, vehicle);
		else
			g_client:getServerConnection():sendEvent(BuyableMFdualsAttachEvent:new(vehicle, attacherVehicle));
		end;
	end;
end;


BuyableMFdualsDetachEvent = {};
BuyableMFdualsDetachEvent_mt = Class(BuyableMFdualsDetachEvent, Event);

InitEventClass(BuyableMFdualsDetachEvent, "BuyableMFdualsDetachEvent");

function BuyableMFdualsDetachEvent:emptyNew()
    local self = Event:new(BuyableMFdualsDetachEvent_mt);
    return self;
end;

function BuyableMFdualsDetachEvent:new(vehicle)
    local self = BuyableMFdualsDetachEvent:emptyNew()
    self.vehicle = vehicle;
    return self;
end;

function BuyableMFdualsDetachEvent:readStream(streamId, connection)
    local id = streamReadInt32(streamId);
    self.vehicle = networkGetObject(id);
    self:run(connection);
end;

function BuyableMFdualsDetachEvent:writeStream(streamId, connection)
    streamWriteInt32(streamId, networkGetObjectId(self.vehicle));
end;

function BuyableMFdualsDetachEvent:run(connection)
	self.vehicle:disassembleTwinWheels(true);
    if not connection:getIsServer() then
        g_server:broadcastEvent(BuyableMFdualsDetachEvent:new(self.vehicle), nil, connection, self.object);
    end;
end;

function BuyableMFdualsDetachEvent.sendEvent(vehicle, noEventSend)
	if noEventSend == nil or noEventSend == false then
		if g_server ~= nil then
			g_server:broadcastEvent(BuyableMFdualsDetachEvent:new(vehicle), nil, nil, vehicle);
		else
			g_client:getServerConnection():sendEvent(BuyableMFdualsDetachEvent:new(vehicle));
		end;
	end;
end;

BuyableMFcageAttachEvent = {};
BuyableMFcageAttachEvent_mt = Class(BuyableMFcageAttachEvent, Event);

InitEventClass(BuyableMFcageAttachEvent, "BuyableMFcageAttachEvent");

function BuyableMFcageAttachEvent:emptyNew()
    local self = Event:new(BuyableMFcageAttachEvent_mt);
    return self;
end;

function BuyableMFcageAttachEvent:new(vehicle, attacherVehicle)
    local self = BuyableMFcageAttachEvent:emptyNew()
    self.vehicle = vehicle;
	self.attacherVehicle = attacherVehicle;
    return self;
end;

function BuyableMFcageAttachEvent:readStream(streamId, connection)
    local id = streamReadInt32(streamId);
	local attacherId = streamReadInt32(streamId);
	self.attacherVehicle = networkGetObject(attacherId);
    self.vehicle = networkGetObject(id);
    self:run(connection);
end;

function BuyableMFcageAttachEvent:writeStream(streamId, connection)
    streamWriteInt32(streamId, networkGetObjectId(self.vehicle));
	streamWriteInt32(streamId, networkGetObjectId(self.attacherVehicle));
end;

function BuyableMFcageAttachEvent:run(connection)
	self.vehicle:assembleCageWheels(self.attacherVehicle, true);
    if not connection:getIsServer() then
        g_server:broadcastEvent(BuyableMFcageAttachEvent:new(self.vehicle, self.attacherVehicle), nil, connection, self.object);
    end;
end;


function BuyableMFcageAttachEvent.sendEvent(vehicle, attacherVehicle, noEventSend)
	if noEventSend == nil or noEventSend == false then
		if g_server ~= nil then
			g_server:broadcastEvent(BuyableMFcageAttachEvent:new(vehicle, attacherVehicle), nil, nil, vehicle);
		else
			g_client:getServerConnection():sendEvent(BuyableMFcageAttachEvent:new(vehicle, attacherVehicle));
		end;
	end;
end;


BuyableMFcageDetachEvent = {};
BuyableMFcageDetachEvent_mt = Class(BuyableMFcageDetachEvent, Event);

InitEventClass(BuyableMFcageDetachEvent, "BuyableMFcageDetachEvent");

function BuyableMFcageDetachEvent:emptyNew()
    local self = Event:new(BuyableMFcageDetachEvent_mt);
    return self;
end;

function BuyableMFcageDetachEvent:new(vehicle)
    local self = BuyableMFcageDetachEvent:emptyNew()
    self.vehicle = vehicle;
    return self;
end;

function BuyableMFcageDetachEvent:readStream(streamId, connection)
    local id = streamReadInt32(streamId);
    self.vehicle = networkGetObject(id);
    self:run(connection);
end;

function BuyableMFcageDetachEvent:writeStream(streamId, connection)
    streamWriteInt32(streamId, networkGetObjectId(self.vehicle));
end;

function BuyableMFcageDetachEvent:run(connection)
	self.vehicle:disassembleCageWheels(true);
    if not connection:getIsServer() then
        g_server:broadcastEvent(BuyableMFcageDetachEvent:new(self.vehicle), nil, connection, self.object);
    end;
end;

function BuyableMFcageDetachEvent.sendEvent(vehicle, noEventSend)
	if noEventSend == nil or noEventSend == false then
		if g_server ~= nil then
			g_server:broadcastEvent(BuyableMFcageDetachEvent:new(vehicle), nil, nil, vehicle);
		else
			g_client:getServerConnection():sendEvent(BuyableMFcageDetachEvent:new(vehicle));
		end;
	end;
end;